# Copyright (c) 2011, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#   NAME
#      s_crsutils.pm
#
#   DESCRIPTION
#      This module contains common OSD functions for root scripts
#
#   NOTES
#      <other useful comments, qualifications, etc.>
#
#   MODIFIED   (MM/DD/YY)
#   sidshank    11/15/12 - fix bug 15843792
#   sidshank    11/11/12 - fix bug 14202806
#   gmaldona    10/25/12 - XbranchMerge gmaldona_bug-14492893 from
#                          st_has_12.1plbeta2_gen
#   minzhu      09/07/12 - add an env-var: SUPPORT_HAIP
#   jmunozn     08/23/12 - Remove s_run_as_user3 and add s_get_qosctl_path
#   gmaldona    08/23/12 - create temporary file in OS TEMP directory
#   ssprasad    07/23/12 - Skip oka files in s_set_ownergroup if they
#                          don't exist.
#   rdasari     07/20/12 - fix bug 14340774
#   shmubeen    07/18/12 - add routines for AFD installation
#   sidshank    07/06/12 - fix 14283643 .
#   nkorehis    06/20/12 - redo fix for bug-12610689
#   sidshank    05/03/12 - remove s_redirect/restore output subroutines
#   xyuan       03/09/12 - Fix bug 13827767
#   sidshank    03/08/12 - Remove dummy routine s_first_node_tasks
#   nkorehis    03/01/12 - bug-12610689:
#   anjiwaji    02/15/12 - Skip acfs files in s_set_onwnergroup if they don't
#                          exist.
#   xyuan       02/02/12 - Modified s_restoreolrloc & s_checkolrbackup
#   xesquive    01/24/12 - Function s_isSUSELinux verifies if is Linux OS
#   gmaldona    01/16/12 - add a method with bidirectional communication
#   xyuan       01/08/12 - Add OSD functions for downgrade
#   xyuan       01/05/12 - Export s_install_initd
#   xyuan       12/25/11 - Use 'crsctl stop crs' for 10.2 & above
#                          in s_stop_OldCrsStack
#   bmanry      12/16/11 - Bug 13506877: Do not unset ORACLE_BASE.
#   gnagiah     11/15/11 - Bug 13058611, keepdg deletes the dg in upgarded env
#   xyuan       08/03/11 - XbranchMerge xyuan_bug-12698968 from
#                          st_has_11.2.0.3.0
#   smatpadi    07/14/11 - Bug/12583050:inittab/upstart error message
#   rdasari     07/06/11 - add EXTSHM=OFF for aix
#   smatpadi    05/20/11 - Fix OEL6 ade startup issue
#   dpham       03/28/11 - Creation for 12c
#
package s_crsutils;

use strict;
use English;
use File::Copy;
use File::Path;
use File::Find;
use File::Basename;
use File::Spec::Functions;
use File::Temp qw/ tempfile /;
use Sys::Hostname;
use Carp;
use Socket;
use Env qw(NLS_LANG);
use Fcntl ':mode';
use POSIX qw(tmpnam);
use Cwd;

# root script modules
use crsutils;

# export vars and functions
use Exporter;
use vars qw(@ISA @EXPORT @EXPORT_OK);

@ISA = qw(Exporter);

my @exp_func = qw(s_check_SuperUser s_set_ownergroup s_reset_crshome
                  s_reset_crshome1 s_set_perms s_osd_setup
                  s_check_CRSConfig s_validate_olrconfig s_get_olr_file
                  s_validate_ocrconfig s_validateOCR s_reset_srvconfig
                  s_register_service s_unregister_service s_check_service
                  s_start_service s_run_as_user s_run_as_user2 s_init_scr
                  s_get_config_key s_isLink s_get_platform_family s_copyOCRLoc
                  s_getOldCrsHome s_redirect_souterr s_restore_souterr
                  s_stop_OldCrsStack s_RemoveInitResources s_CleanTempFiles
                  s_setParentDirOwner s_resetParentDirOwner s_checkOracleCM
                  s_ResetOLR s_ResetOCR s_ResetVotedisks s_createConfigEnvFile
                  s_isRAC_appropriate s_createLocalOnlyOCR s_is92ConfigExists
                  s_configureCvuRpm s_removeCvuRpm s_remove_file s_getAbsLink
                  s_removeGPnPprofile s_crf_check_bdbloc s_crf_remove_itab
                  s_is_HAIP_supported s_is_HAIP_NonFatal s_CheckNetworkConfig
                  s_houseCleaning s_NSCleanUp s_install_initd s_checkolrbackup
                  s_restoreInitScripts s_restoreASMFiles s_restoreolrloc
                  s_rm_afdinit_init s_rm_afdinit_rclevel s_rm_afdinit_rclevel
                  s_get_qosctl_path
                 );

push @EXPORT, @exp_func;

my ($ARCHIVE, $INITD, $ENVMT);
my @ns_dir            = ("/var/tmp/.oracle","/tmp/.oracle");
my @ns_files          = ("CSS","CRS","EVM","PROC","css","crs","evm","proc");
my $dev_null          = "/dev/null";
my $FSSEP             = "/";
my $oelNetworkConfig  = "/etc/sysconfig/network";
my $suseNetworkConfig = "/etc/sysconfig/network/config";
my $checkNetworkTemp  = ".orcl.tmp.$$";
my $checkNetworkSave  = ".orcl.$$";

# Linux Upstart definitions and globals
my $UPSTART_OHASD_SERVICE   = "oracle-ohasd";
my $UPSTART_OHASD_CONF_FILE = "oracle-ohasd.conf";
my $RESTORECON              = "/sbin/restorecon";
my $UPSTART_USED            = -1;

if ($OSNAME eq 'linux') {
   $INITD   = '/etc/init.d';
   $ARCHIVE = '/usr/bin/ar';
   $ENVMT   = '/usr/bin/env';
} elsif ($OSNAME eq 'solaris') {
   $INITD   = '/etc/init.d';
   $ARCHIVE = '/usr/ccs/bin/ar';
   $ENVMT   = '/usr/bin/env';
} elsif ($OSNAME eq 'hpux') {
   $INITD   = '/sbin/init.d';
   $ARCHIVE = '/usr/ccs/bin/ar';
   $ENVMT   = "/bin/env";
} elsif ($OSNAME eq 'aix') {
   $INITD   = '/etc';
   $ARCHIVE = '/usr/ccs/bin/ar -X64';
   $ENVMT   = '/usr/bin/env';
} elsif ($OSNAME eq 'dec_osf') {
   $INITD   = '/sbin/init.d';
   $ARCHIVE = '/usr/bin/ar';
   $ENVMT   = '/usr/bin/env';
}

####---------------------------------------------------------
#### Function for checking and returning Super User name
# ARGS : 0
sub s_check_SuperUser
{
   my $superUser = "root";
   my $program   = "this script";

   # get user-name
   my $usrname = getpwuid ($<);
   if ($usrname ne $superUser) {
      print_error(47);
      trace ("You must be logged in as $superUser to run $program.");
      trace ("Log in as $superUser and rerun $program.");
      return "";
   }

   return $superUser;
}

####---------------------------------------------------------
#### Function for setting user and group on a specified path
# ARGS : 3
# ARG1 : Oracle owner
# ARG2 : Oracle group 
# ARG3 : file
sub s_set_ownergroup
{
   my ($owner, $group, $file) = @_;

   if (!$owner) {
      print_error(40);
      return FAILED;
   }

   if (!$group) {
      print_error(41);
      return FAILED;
   }

   if (!$file) {
      print_error(42);
      return FAILED;
   }

   if (!(-e $file)) {
      # Skip missing ACFS files 
      if ( $file=~ m/.*acfs.*/ )
      {
         return SUCCESS; 
      }
      elsif ( $file=~ m/.*oka.*/ )
      {
         # Skip missing OKA files 
         return SUCCESS; 
      }
      elsif ( $file=~ m/.*afd.*/ )
      {
         return SUCCESS; 
      }
      else
      {    
         print_error(46, $file);
         return FAILED;
      }
   }

   my $uid = getpwnam ($owner);
   my $gid = getgrnam ($group);
   if ($CFG->DEBUG) {
      trace("Setting owner ($owner:$uid) and group ($group:$gid) on file $file");
   }

   if (! chown ($uid, $gid, $file)) {
      print_error(152, $file);
      trace ("Can't change ownership of $file: $!");
      return FAILED;
   }

   return SUCCESS;
}

####---------------------------------------------------------
#### Function for resetting owner and permissions of CRS home dirs/files
# ARGS : 4
# ARG1 : Oracle owner
# ARG2 : Oracle group
# ARG3 : perms
# ARG4 : directory path
sub s_reset_crshome
{
   my ($owner, $group, $perms, $basedir) = @_;
   my $exclfile = catfile($CFG->ORA_CRS_HOME, 'crs', 'install', 'install.excl');

   s_reset_crshome1($owner, $group, $perms, $basedir, $exclfile);
}

####---------------------------------------------------------
#### Function for resetting owner and permissions of CRS home dirs/files
# ARGS : 5
# ARG1 : Oracle owner
# ARG2 : Oracle group
# ARG3 : perms
# ARG4 : directory path
# ARG5 : exclude file path
sub s_reset_crshome1
{
   if (is_dev_env()) {
      return SUCCESS;
   }

   my ($owner, $group, $perms, $basedir, $exclfile) = @_;

   if (!$owner) {
      print_error(40);
      return FAILED;
   }

   if (!$group) {
      print_error(41);
      return FAILED;
   }

   my $userid   = getpwnam ($owner);
   my $groupid  = getgrnam ($group);
   my @excl_dir = ();

   if (-e $exclfile) {
      @excl_dir = read_file (catfile($exclfile));
   }

   # reset owner/group of directory/file name of basedir 
   # and its parent dir to ORACLE_OWNER/DBA
   finddepth (\&reset_perms, $basedir);
   my @parent;
   $parent[0]=dirname($basedir);
   foreach(@parent) {
           &reset_perms;
   }

   sub reset_perms
   {
      #Just give write permissions to grid owner for all files
      #during reset and exclude links .
      if (! -l $_) {
         my ($dev,$ino,$mode) = lstat($_);
         $perms               = S_IMODE($mode);
         my $origperm         = sprintf "%04o", $perms;
         $perms               = $perms | 128;
         my $newperm          = sprintf "%04o", $perms;

         if (SetPerms()) {
            if ($CFG->DEBUG) {
               trace("orig perm for $_ is $origperm, setting file perm to $newperm");
            }

	    chown ($userid, $groupid, $_);
	    chmod (oct ($newperm), $_);
         }
      }
   }

   sub SetPerms
   {
      if (! $CFG->UNLOCK) {
         return TRUE;
      }

      # if UNLOCK, bypass files that are in exclude dir
      my $setperms = TRUE;
      foreach my $dir (@excl_dir) {
         chomp ($dir);
         next if ($dir =~ /^#|^\s*$/);  # skip blanks and comments

	 # do not set perms if file is on execluded dir
         if ($File::Find::dir =~ /$dir/) {
            $setperms = FALSE;
         }
      }

      return $setperms;
   }

   return SUCCESS;
}

####---------------------------------------------------------
#### Function for setting permissions on a specified path
# ARGS : 2
# ARG1 : permissions
# ARG3 : file/dir
sub s_set_perms
{
   my ($perms, $file) = @_;

   if (!$perms) {
      print_error(43);
      return FAILED;
   }

   if (!$file) {
      print_error(42);
      return FAILED;
   }

   if (!(-e $file)) {
      print_error(46, $file);
      return "FAILED";
   }

   if ($CFG->DEBUG) { trace ("Setting permissions ($perms) on file/dir $file"); }

   if ($CFG->SIHA) {
      if (is_dev_env()) {
         if (! chmod (oct($perms), $file)) {
            print_error(153, $file);
            trace("Can't change permissions of $file: $!");
            return FAILED;
         }
      }
      elsif (! chmod (((oct($perms)) & oct(4770)), $file)) {
         print_error(153, $file);
         trace("Can't change permissions of $file: $!");
         return FAILED;
      }
   }
   elsif (! chmod (oct($perms), $file)) {
      print_error(153, $file);
      trace("Can't change permissions of $file: $!");
      return FAILED;
   }

   return SUCCESS;
}

####---------------------------------------------------------
#### Functions for copying script to init directory
# ARGS : 2
# ARG1 : init script name
# ARG2 : destination file
sub s_copy_to_initdir
{
   my $sourcefile = $_[0];
   my $destfile   = $_[1];

   if (!$sourcefile) {
      print_error(44);
      return FAILED;
   }

   if (!(-f $sourcefile)) {
      print_error(45, $sourcefile);
      return FAILED;
   }

   trace ("init file = " . $sourcefile);
   trace ("Copying file " . $sourcefile . " to " . $CFG->params('ID') . 
          " directory");
   copy ($sourcefile, catfile ($CFG->params('ID'), $destfile)) || return FAILED;

   trace ("Setting " . $destfile . " permission in " . $CFG->params('ID') . 
          " directory");
   s_set_perms ("0755", catfile ($CFG->params('ID'), $destfile)) || return FAILED;

   return SUCCESS;
}

####---------------------------------------------------------
#### Functions for copying script to init directory
# ARGS : 1
# ARG1 : init script name
# ARG1 : dest name
sub s_copy_to_rcdirs
{
   my $sourcefile = $_[0];
   my $destfile   = $_[1];

   if (!$sourcefile) {
      print_error(44);
      return FAILED;
   }

   if (!(-f $sourcefile)) {
      print_error(45, $sourcefile);
      return FAILED;
   }

   trace ("init file = " . $sourcefile);

   # Copy to init dir
   trace ("Copying file " . $sourcefile . " to " . 
          $CFG->params('ID') . " directory");
   copy ($sourcefile, catfile ($CFG->params('ID'), $destfile)) || return FAILED;
   trace ("Setting " . $destfile . " permission in " .
          $CFG->params('ID') . " directory");
   s_set_perms ("0755", catfile ($CFG->params('ID'), $destfile)) || return FAILED;
    if (s_isSUSELinux()) {
       # for SUSE Linux, do not create link to the file in the init dir
       return SUCCESS;
    }

   # Create a link to the file in the init dir
   my @RCSDIRLIST = split (/ /, $CFG->params('RCSDIR'));
   my $RC_START   = $CFG->params('RC_START');
   foreach my $rc (@RCSDIRLIST) {
      trace ("Removing \"" . $rc . "/" . $RC_START . $destfile . "\"");
      s_remove_file ("$rc/$RC_START$destfile");

      trace ("Creating a link \"" . catfile ($rc, "$RC_START$destfile") .
             "\" pointing to " . catfile ($CFG->params('ID'), $destfile));
      symlink (catfile($CFG->params('ID'), $destfile),
               catfile($rc, "$RC_START$destfile")) || return FAILED;
   }
   my @RCKDIRLIST = split (/ /, $CFG->params('RCKDIR'));
   my $RC_KILL    = $CFG->params('RC_KILL');
   foreach my $rc (@RCKDIRLIST) {
      trace ("Removing \"" . $rc . "/" . $RC_KILL . $destfile . "\"");
      s_remove_file ("$rc/$RC_KILL$destfile");

      trace ("Creating a link \"" . catfile ($rc, "$RC_KILL$destfile") .
             "\" pointing to " . catfile ($CFG->params('ID'), $destfile));
      symlink(catfile($CFG->params('ID'), $destfile),
              catfile($rc, "$RC_KILL$destfile")) || return FAILED;
   }

   trace ("The file " . $destfile .
          " has been successfully linked to the RC directories");
   return SUCCESS;
}

sub s_NSCleanUp
{
   trace ("Cleaning up Network socket directories");

   foreach my $nsdir (@ns_dir) {
      foreach my $file (<$nsdir/*>) {
         foreach my $ns (@ns_files) {
            if ($file =~ $ns) {
               trace("Unlinking file : $file");
               unlink($file);
            }
         }
      }
   }
}

####---------------------------------------------------------
#### Functions for removing script from rc directories
# ARGS : 1
# ARG1 : init script name
sub s_clean_rcdirs
{
   my $file = $_[0];

   if (!$file) {
      print_error(44);
      trace ("Null value passed for init script file name");
      return FAILED;
   }

   trace ("Init file = " . $file);
   trace ("Removing \"" . $file . "\" from RC dirs");

   #remove old ones
   if ($CFG->params('RCALLDIR')) {
      my ($rc, $rcStartFile, $rcKillFile, $rcKillOldFile, $rcKillOld2File);
      my @RCALLDIRLIST = split (/ /, $CFG->params('RCALLDIR'));

      foreach $rc (@RCALLDIRLIST) {
         if ($CFG->params('RC_START')) {
            $rcStartFile = catfile ($rc, $CFG->params('RC_START') . $file);
            s_remove_file ("$rcStartFile");
         }

         if ($CFG->params('RC_KILL')) {
            $rcKillFile = catfile ($rc, $CFG->params('RC_KILL') . $file);
            s_remove_file ("$rcKillFile");
         }

         if ($CFG->params('RC_KILL_OLD')) {
            $rcKillOldFile = catfile ($rc, $CFG->params('RC_KILL_OLD') . $file);
            s_remove_file ("$rcKillOldFile");
         }

         if ($CFG->defined_param('RC_KILL_OLD2')) {
            $rcKillOld2File = catfile ($rc, $CFG->params('RC_KILL_OLD2') . $file);
            s_remove_file ("$rcKillOld2File");
         }
      }
   }
}

####---------------------------------------------------------
#### Function for adding CRS entries to inittab
# ARGS : 0
sub s_add_itab
{
   # If upstart is being used, then add the upstart conf file.
   if (s_is_Linux_Upstart()) 
   {
     return (s_add_upstart_conf());
   }

   my $INITTAB_CH = catfile ($CFG->ORA_CRS_HOME, "crs", "install", "inittab");

   if (-e $INITTAB_CH) {
      my $IT = $CFG->params('IT');
      unless (open (ITAB, "<$IT")) {
         print_error(206, $IT, $!);
         return FAILED;
      }

      unless (open (ITABNOCRS, ">$IT.no_crs")) {
         print_error(207, "$IT.no_crs", $!);
         return FAILED;
      }

      # Remove ohasd from inittab
      while (<ITAB>) {
         if (!($_ =~ /init.ohasd/)) {
            print ITABNOCRS "$_";
         }
      }

      close (ITABNOCRS);
      close (ITAB);

      trace("Created backup $IT.no_crs");
      unless (copy ("$IT.no_crs", "$IT.tmp")) {
         print_error(105, "$IT.no_crs", "$IT.tmp", $!);
         return FAILED;
      }

      unless (open (CRSITAB, "<$INITTAB_CH")) {
         print_error(206, $INITTAB_CH, $!);
         return FAILED;
      }

      unless (open (ITABTMP, ">>$IT.tmp")) {
         print_error(208, "$IT.tmp", $!);
         return FAILED;
      }

      trace ("Appending to $IT.tmp:");
      while (<CRSITAB>) {
         print ITABTMP "\n";
         print ITABTMP "$_";
         trace ("$_");
      }
      
      close (ITABTMP);
      close (CRSITAB);

      trace ("Done updating $IT.tmp");
      unless (copy ("$IT.tmp", "$IT.crs")) {
         print_error(105, "$IT.tmp", "$IT.crs", $!);
         return FAILED;
      }
      
      trace("Saved $IT.crs");
      unless (move ("$IT.tmp", $IT)) {
         print_error(209, "$IT.tmp", $IT, $!);
         return FAILED;
      }
      
      trace("Installed new $IT");
   }
   else {
      print_error(13, $INITTAB_CH);
      return FAILED;
   }

   return SUCCESS;
}

####---------------------------------------------------------
#### Function for removing CRS entries from inittab
# ARGS : 1 - match pattern for inittab entries
sub s_remove_itab
{
   my $match_pattern = $_[0];
   my $IT            = $CFG->params('IT');
   trace ("itab entries=$match_pattern");

   # If upstart is being used, then remove the conf files
   if (s_is_Linux_Upstart())
   {
      return (s_remove_upstart_conf($match_pattern));
   }

   unless (open (ITAB, "<$IT")) {
      print_error(206, $IT, $!);
      return FAILED;
   }
   
   unless (open (ITABTMP, ">$IT.tmp")) {
      print_error(207, $IT, $!);
      return FAILED;
   }

   while (<ITAB>) {
      if (!($_ =~ /init.($match_pattern)/)) {
         print ITABTMP "$_";
      }
   }
   
   close (ITABTMP);
   close (ITAB);

   unless (copy ("$IT.tmp", "$IT.no_crs")) {
      print_error(105, "$IT.tmp", "$IT.no_crs", $!);
      return FAILED;
   }

   unless (move ("$IT.tmp", $IT)) {
      print_error(209, "$IT.tmp", $IT, $!);
      return FAILED;
   }

   return SUCCESS;
}

####-----------------------------------------------------------------------
#### Function for checking if CRS is already configured
# ARGS: 2
# ARG1: hostname
# ARG2: crs user
sub s_check_CRSConfig
{
    my $hostname  = $_[0];
    my $crsuser   = $_[1];
    my $FATALFILE = catfile($CFG->params('SCRBASE'),
                            $hostname, $crsuser, "cssfatal");
    if ((-f $FATALFILE) && (-f $CFG->params('OCRCONFIG'))) {
       trace ("Oracle CRS stack is already configured and will be " .
              "running under init \(1M\)");
       return TRUE;
    }
    else {
       trace ("Oracle CRS stack is not configured yet");
       return FALSE;
   }
}

####-----------------------------------------------------------------------
#### Function for validating olr.loc file and creating it if does not exist
# ARGS: 2
# ARG1 : Complete path of OLR location
# ARG2 : CRS Home
sub s_validate_olrconfig
{
   my $olrlocation = $_[0];
   my $crshome     = $_[1];
   my $OLRCONFIG   = $CFG->params('OLRCONFIG');

   trace ("Validating " . $CFG->params('OLRCONFIG') .
          " file for OLR location " . $olrlocation);

   ## @todo Check existing olr.loc file. If it exists, then check value of
   #  olrconfig_loc property. If it's same as the one passed on the call
   #  then go ahead. Else, throw an error msg and quit the installation.
   if (-f $OLRCONFIG) {
      trace ("$OLRCONFIG already exists. Backing up " . $OLRCONFIG .
             " to " . $OLRCONFIG . ".orig");
      # Need to remove this once the @todo is implemented.
      copy ($OLRCONFIG, $OLRCONFIG . ".orig") || return FAILED;
   }

   open (OLRCFGFILE, ">$OLRCONFIG") || return FAILED;
   print OLRCFGFILE "olrconfig_loc=$olrlocation\n";
   print OLRCFGFILE "crs_home=$crshome\n";
   close (OLRCFGFILE);

   #FIXME: This should be moved to add_olr_ocr_vdisks_locs
   if (is_dev_env()) {
      s_set_ownergroup($CFG->params('ORACLE_OWNER'),
                       $CFG->params('ORA_DBA_GROUP'), 
                       $CFG->params('OLRCONFIG'));
   } 
   else {
      s_set_ownergroup($CFG->SUPERUSER,
                       $CFG->params('ORA_DBA_GROUP'), 
                       $CFG->params('OLRCONFIG'));
   }

   s_set_perms ("0644", $CFG->params('OLRCONFIG'));
   trace("Done setting permissions on file " . $CFG->params('OLRCONFIG'));
   return SUCCESS;
}

sub s_get_olr_file
#-------------------------------------------------------------------------------
# Function:  Get key from olr.loc
# Args    :  1 - key
# Returns :  Key's value
#-------------------------------------------------------------------------------
{
   my $key       = $_[0];
   my $ret       = "";
   my $OLRCONFIG = $CFG->params('OLRCONFIG');

   if (!(-r $OLRCONFIG)) {
      # Don't show up a NLS error message here because we also
      # call this function when upgrade from 10.2, and the $OLRCONFIG is
      # not there in this scenario.
      trace ("Either " . $OLRCONFIG . " does not exist or is not readable");
      trace ("Make sure the file exists and it has read and execute access");
      return $ret;
   }

   open (OLRCFGFILE, "<$OLRCONFIG") || return $ret;

   while (<OLRCFGFILE>) {
      if (/^$key=(\S+)/) {
         $ret = $1;
         last;
      }
   }
   
   close (OLRCFGFILE);
   return $ret;
}

####---------------------------------------------------------
#### Function for validating ocr.loc file
# ARGS: 2
# ARG1 : ocrlocations
# ARG2 : isHas
sub s_validate_ocrconfig
{
   my $ocrlocations = $_[0];
   my $isHas        = $_[1];
   my $OCRCONFIG    = $CFG->params('OCRCONFIG');

   trace ("Validating OCR locations in " . $CFG->params('OCRCONFIG'));
   trace ("Checking for existence of " . $CFG->params('OCRCONFIG'));

   if (-f $OCRCONFIG) {
      trace ("Backing up " . $OCRCONFIG . " to " . $OCRCONFIG . ".orig");
      copy ($OCRCONFIG, $OCRCONFIG . ".orig") || return FAILED;
   }

   my ($ocrlocation,
       $ocrmirrorlocation,
       $ocrlocation3,
       $ocrlocation4,
       $ocrlocation5) = split (/\s*,\s*/, $ocrlocations);

   open (OCRCFGFILE, ">$OCRCONFIG") || return FAILED;

   trace ("Setting ocr location " . $ocrlocation);
   print OCRCFGFILE "ocrconfig_loc=$ocrlocation\n";

   if ($ocrmirrorlocation) {
      trace ("Setting ocr mirror location " . $ocrmirrorlocation);
      print OCRCFGFILE "ocrmirrorconfig_loc=$ocrmirrorlocation\n";
   }

   if ($ocrlocation3) {
      trace ("Setting ocr location3 " . $ocrlocation3);
      print OCRCFGFILE "ocrconfig_loc3=$ocrlocation3\n";
   }

   if ($ocrlocation4) {
      trace ("Setting ocr location4 " . $ocrlocation4);
      print OCRCFGFILE "ocrconfig_loc4=$ocrlocation4\n";
   }

   if ($ocrlocation5) {
      trace ("Setting ocr location5 " . $ocrlocation5);
      print OCRCFGFILE "ocrconfig_loc5=$ocrlocation5\n";
   }

   if ($isHas) {
      print OCRCFGFILE "local_only=TRUE\n";
   }
    else {
      print OCRCFGFILE "local_only=FALSE\n";
   }

   close (OCRCFGFILE);
   return SUCCESS;
}

####---------------------------------------------------------
#### Validating OCR locations based on existing ocr settings
# ARGS: 3
# ARG1 : Path for Oracle CRS home
# ARG2 : Cluster name
# ARG3 : Comma separated OCR locations
sub s_validateOCR
{
    my $crshome        = $_[0];
    my $clustername    = $_[1];
    my $ocrlocations   = $_[2];
    my $status         = SUCCESS;
    my $OCR_SYNC_FILE  = catfile ($crshome, "srvm", "admin", $CFG->params('OCRLOC'));
    my $OCR_CDATA_DIR  = catfile ($crshome, "cdata");
    my $OCR_BACKUP_DIR = catfile ($crshome, "cdata", $clustername);

    my $SRVCONFIG_LOC = "";
    if (-f $CFG->params('SRVCONFIG')) {
        trace ("Checking repository used for 9i installations");

        # srvConfig.loc file exists and repository location is
        $SRVCONFIG_LOC = s_get_config_key ("srv", "srvconfig_loc");

        if ($SRVCONFIG_LOC eq "/dev/null") {
            # 9.x srvconfig_loc is already invalidated. So ignore it
            # # take the location entered by user to populate ocr.loc
            $SRVCONFIG_LOC = "";
        }
    }

    ##Checking the OCR locations used by 10gR1 or previous 10gR2
    #installations
    my $OCRCONFIG_LOC = "";
    my $OCRMIRRORCONFIG_LOC = "";
    my $OCRCONFIG_LOC3 = "";
    my $OCRCONFIG_LOC4 = "";
    my $OCRCONFIG_LOC5 = "";

    if (-f $CFG->params('OCRCONFIG')) {
        trace ("Retrieving OCR location used by previous installations");
        # ocr.loc file exists and ocr location set here is
        $OCRCONFIG_LOC       = s_get_config_key ("ocr", "ocrconfig_loc");
        $OCRMIRRORCONFIG_LOC = s_get_config_key ("ocr", "ocrmirrorconfig_loc");
        $OCRCONFIG_LOC3      = s_get_config_key ("ocr", "ocrconfig_loc3");
        $OCRCONFIG_LOC4      = s_get_config_key ("ocr", "ocrconfig_loc4");
        $OCRCONFIG_LOC5      = s_get_config_key ("ocr", "ocrconfig_loc5");
    }

    my $OCRFILE = $CFG->params('OCRCONFIG');

    trace ("Checking if OCR sync file exists");

    if (-f $OCR_SYNC_FILE) {
        trace ("$OCR_SYNC_FILE exists");
        ##Checking the OCR locations used by existing nodes in the cluster
        my $NEW_OCR_FILE = "";
        my $NEW_OCRMIRROR_FILE = "";
        my $NEW_OCRMIRROR_LOC3 = "";
        my $NEW_OCRMIRROR_LOC4 = "";
        my $NEW_OCRMIRROR_LOC5 = "";

        open (OCRSYNCFILE, "<$OCR_SYNC_FILE");

        while (<OCRSYNCFILE>) {
            if (/^ocrconfig_loc=(\S+)/) {
                $NEW_OCR_FILE = $1;
            }
            if (/^ocrmirrorconfig_loc=(\S+)/) {
                $NEW_OCRMIRROR_FILE = $1;
            }
            if (/^ocrconfig_loc3=(\S+)/) {
                $NEW_OCRMIRROR_LOC3 = $1;
            }
            if (/^ocrconfig_loc4=(\S+)/) {
                $NEW_OCRMIRROR_LOC4 = $1;
            }
            if (/^ocrconfig_loc5=(\S+)/) {
                $NEW_OCRMIRROR_LOC5 = $1;
            }
        }
        close (OCRSYNCFILE);

        trace ("NEW_OCR_FILE=$NEW_OCR_FILE");
        trace ("NEW_OCRMIRROR_FILE=$NEW_OCRMIRROR_FILE");
        trace ("NEW_OCRMIRROR_LOC3=$NEW_OCRMIRROR_LOC3");
        trace ("NEW_OCRMIRROR_LOC4=$NEW_OCRMIRROR_LOC4");
        trace ("NEW_OCRMIRROR_LOC5=$NEW_OCRMIRROR_LOC5");

        $ocrlocations = $NEW_OCR_FILE;

        if ($NEW_OCRMIRROR_FILE) {
            $ocrlocations = "$ocrlocations,$NEW_OCRMIRROR_FILE";
        }
        if ($NEW_OCRMIRROR_LOC3) {
            $ocrlocations = "$ocrlocations,$NEW_OCRMIRROR_LOC3";
        }
        if ($NEW_OCRMIRROR_LOC4) {
            $ocrlocations = "$ocrlocations,$NEW_OCRMIRROR_LOC4";
        }
        if ($NEW_OCRMIRROR_LOC5) {
            $ocrlocations = "$ocrlocations,$NEW_OCRMIRROR_LOC5";
        }

        trace ("OCR locations (obtained from $OCR_SYNC_FILE) = $ocrlocations");
    } else {
        ##Syncing of OCR disks is not required
        trace ("No need to sync OCR file");
    }

    my ($OCR_LOCATION,$OCR_MIRROR_LOCATION,$OCR_MIRROR_LOC3,
	$OCR_MIRROR_LOC4,$OCR_MIRROR_LOC5) = split (/\s*,\s*/, $ocrlocations);

    trace ("OCR_LOCATION=$OCR_LOCATION");
    trace ("OCR_MIRROR_LOCATION=$OCR_MIRROR_LOCATION");
    trace ("OCR_MIRROR_LOC3=$OCR_MIRROR_LOC3");
    trace ("OCR_MIRROR_LOC4=$OCR_MIRROR_LOC4");
    trace ("OCR_MIRROR_LOC5=$OCR_MIRROR_LOC5");
    trace ("Current OCR location= $OCRCONFIG_LOC");
    trace ("Current OCR mirror location= $OCRMIRRORCONFIG_LOC");
    trace ("Current OCR mirror loc3=$OCRCONFIG_LOC3");
    trace ("Current OCR mirror loc4=$OCRCONFIG_LOC4");
    trace ("Current OCR mirror loc5=$OCRCONFIG_LOC5");
    trace ("Verifying current OCR settings with user entered values");

    if ($OCRCONFIG_LOC) {
        if ($OCR_LOCATION ne $OCRCONFIG_LOC) {
            my $cfgfile = catfile($crshome, "crs", "install", 
                                   "crsconfig_params");
            print_error(210, $OCRCONFIG_LOC, $OCRFILE, $OCR_LOCATION);
            print_error(212, $OCRFILE, $OCR_LOCATION, $cfgfile, $OCRCONFIG_LOC);
            trace ("Update either \"$OCRFILE\" to use \"$OCR_LOCATION\" or " .
                   "variable OCR_LOCATIONS property set in " .
                   catfile ($crshome, "crs", "install", "crsconfig_params") .
                "with \"$OCRCONFIG_LOC\" then rerun this script");
            return FAILED;
        }
    } else {
        #set ocrconfig_loc = OCR_LOCATION
        $OCRCONFIG_LOC = $OCR_LOCATION;
    }

    if ($OCRMIRRORCONFIG_LOC) {
        if ($OCR_MIRROR_LOCATION ne $OCRMIRRORCONFIG_LOC) {
            my $cfgfile = catfile($crshome, "crs", "install", "crsconfig_params");
            print_error(211, $OCRMIRRORCONFIG_LOC, $OCRFILE, 
                         $OCR_MIRROR_LOCATION);
            print_error(212, $OCRFILE, $OCR_MIRROR_LOCATION, $cfgfile,
                         $OCRMIRRORCONFIG_LOC); 
            trace ("Update either \"$OCRFILE\" to use " .
                   "\"$OCR_MIRROR_LOCATION\" or variable OCR_LOCATIONS " .
                   "property set in " .
                   catfile ($crshome, "crs" . "install" . "crsconfig_params") .
                   " with \"$OCRMIRRORCONFIG_LOC\" then rerun this script");
            return FAILED;
        }
    } else {
        #set the mirror location=user entered value for OCR_MIRROR_LOCATION
        $OCRMIRRORCONFIG_LOC = $OCR_MIRROR_LOCATION;
    }

    if ($OCRCONFIG_LOC3) {
        if ($OCR_MIRROR_LOC3 ne $OCRCONFIG_LOC3) {
            my $cfgfile = catfile($crshome, "crs", "install",
                                   "crsconfig_params");
            print_error(211, $OCRCONFIG_LOC3, $OCRFILE, $OCR_MIRROR_LOC3);
            print_error(212, $OCRFILE, $OCR_MIRROR_LOC3, $cfgfile, 
                         $OCRCONFIG_LOC3);
            trace ("Update either \"$OCRFILE\" to use " .
                   "\"$OCR_MIRROR_LOC3\" or variable OCR_LOCATIONS " .
                   "property set in " .
                   catfile ($crshome, "crs" . "install" . "crsconfig_params") .
                   " with \"$OCRCONFIG_LOC3\" then rerun this script");
            return FAILED;
        }
    } else {
        #set the mirror location=user entered value for OCR_MIRROR_LOCATION
        $OCRCONFIG_LOC3 = $OCR_MIRROR_LOC3;
    }

    if ($OCRCONFIG_LOC4) {
        if ($OCR_MIRROR_LOC4 ne $OCRCONFIG_LOC4) {
            my $cfgfile = catfile($crshome, "crs", "install",
                                   "crsconfig_params");
            print_error(211, $OCRCONFIG_LOC4, $OCRFILE, $OCR_MIRROR_LOC4);
            print_error(212, $OCRFILE, $OCR_MIRROR_LOC4, $cfgfile,
                         $OCRCONFIG_LOC4);
            trace ("Update either \"$OCRFILE\" to use " .
                   "\"$OCR_MIRROR_LOC4\" or variable OCR_LOCATIONS " .
                   "property set in " .
                   catfile ($crshome, "crs" . "install" . "crsconfig_params") .
                   " with \"$OCRCONFIG_LOC4\" then rerun this script");
            return FAILED;
        }
    } else {
        #set the mirror location=user entered value for OCR_MIRROR_LOCATION
        $OCRCONFIG_LOC4 = $OCR_MIRROR_LOC4;
    }

    if ($OCRCONFIG_LOC5) {
        if ($OCR_MIRROR_LOC5 ne $OCRCONFIG_LOC5) {
            my $cfgfile = catfile($crshome, "crs", "install", "crsconfig_params");
            print_error(211, $OCRCONFIG_LOC5, $OCRFILE, $OCR_MIRROR_LOC5);
            print_error(212, $OCRFILE, $OCR_MIRROR_LOC5, $cfgfile,
                          $OCRCONFIG_LOC5);
            trace ("Update either \"$OCRFILE\" to use " .
                   "\"$OCR_MIRROR_LOC5\" or variable OCR_LOCATIONS " .
                   "property set in " .
                   catfile ($crshome, "crs" . "install" . "crsconfig_params") .
                   " with \"$OCRCONFIG_LOC5\" then rerun this script");
            return FAILED;
        }
    } else {
        #set the mirror location=user entered value for OCR_MIRROR_LOCATION
        $OCRCONFIG_LOC5 = $OCR_MIRROR_LOC5;
    }

    trace ("Setting OCR locations in " . $CFG->params('OCRCONFIG'));
    s_validate_ocrconfig ($ocrlocations, 0) or (return FAILED);

    if (-f $OCR_SYNC_FILE) {
        trace ("Removing OCR sync file: $OCR_SYNC_FILE");
        s_remove_file ("$OCR_SYNC_FILE");
    }

    return $status;
}

####---------------------------------------------------------
#### Function for invalidating srvconfig_loc in srvconfig.loc file
sub s_reset_srvconfig
{
   # Invalidate the existing srvConfig.loc file if it was existing 
   trace ("Invalidating repository location for Oracle 9i deployments");

   if (-f $CFG->params('SRVCONFIG')) {
      my $SRVCONFIG = $CFG->params('SRVCONFIG');
      open (SRVCFGFILE, ">$SRVCONFIG") || return FAILED;
      print SRVCFGFILE "srvconfig_loc=/dev/null\n";
      close (SRVCFGFILE);

      s_set_ownergroup($CFG->SUPERUSER,
                       $CFG->params('ORA_DBA_GROUP'), 
                       $CFG->params('SRVCONFIG')) || return FAILED;
      s_set_perms("0644", $CFG->params('SRVCONFIG')) || return FAILED;
   }

   return SUCCESS;
}

####---------------------------------------------------------
#### Function for registering daemon/service with init
# ARGS: 1
# ARG1: daemon to be registered
sub s_register_service
{
    my $srv = $_[0];

    # Setup init scripts
    my $INITDIR = catfile ($CFG->params('ORACLE_HOME'), "crs", "init");
    my $INITDIR_INITSRV = catfile ($INITDIR, "init.$srv");
    my $INITDIR_SRV; 
    #if SUSE linux, copy the ohasd.sles to the rcdirs
    if (! s_isSUSELinux()) {
       $INITDIR_SRV = catfile ($INITDIR, $srv);
    } else {
       $INITDIR_SRV = catfile ($INITDIR, "$srv.sles");
    }
    
    s_copy_to_initdir($INITDIR_INITSRV, "init.$srv") || return FAILED;
    s_copy_to_rcdirs ($INITDIR_SRV, $srv) || return FAILED;

    return SUCCESS;
}

#### Function for starting daemon/service
# ARGS: 3
# ARG1: daemon to be started
# ARG2: user under whom daemon/service needs to be started
sub s_start_service
{
   my $srv  = $_[0];
   my $user = $_[1];
   my $crsctl = crs_exec_path('crsctl');
   my ($status, $grep_val);
   my @output; 
   my @cmdout;

   # Check to see if the service is OHASD
   if (($srv eq "ohasd") || ($srv eq "crsexcl")) {
      # Create the autorun file and its directory
      my $dir = catdir ($CFG->params('SCRBASE'), $CFG->HOST, $CFG->HAS_USER);
      create_dir($dir);
      
      my $AUTORUNFILE = catfile ($dir, "ohasdrun");
      open (AUTORUN, ">$AUTORUNFILE") || die(dieformat(207, $AUTORUNFILE, $!));
      print AUTORUN "stop\n";
      close (AUTORUN);

      s_set_ownergroup($CFG->HAS_USER, 
                       $CFG->HAS_GROUP,
                       $AUTORUNFILE) || die(dieformat(152, $AUTORUNFILE));
      s_set_perms ("0644", $AUTORUNFILE) || die(dieformat(153, $AUTORUNFILE));

      my $INIT_FILE = $CFG->params('IT');
      if (s_is_Linux_Upstart())
      {
           $INIT_FILE = "$UPSTART_OHASD_CONF_FILE";
      }

      # Add OHASD to inittab/upstart
      if ($CFG->DOWNGRADE) { 
         print_error(328, $INIT_FILE);
      } 
      elsif($CFG->UPGRADE) {
         print_error(329, $INIT_FILE);
      }
      else {
         print_error(330, $INIT_FILE);
      }

      s_remove_itab ("cssd|evmd|crsd|ohasd") || return FAILED;
      if ( ! s_is_Linux_Upstart()) {
         $status = system ($CFG->params('INIT') . " q");
         if (0 != $status) {
            print_error(213, $INIT_FILE, $!);
            return FAILED;
         }
      }

      sleep (5);
      s_add_itab() || return FAILED;

      if (s_is_Linux_Upstart()) {
         my $INITCTL = $CFG->params('INITCTL');
         ($status, @output) = system_cmd_capture("$INITCTL start $UPSTART_OHASD_SERVICE");
         if (0 != $status) {
            # Failed to start $UPSTART_OHASD_SERVICE
            print_error(213, $INIT_FILE, $!);
            return FAILED;
         }
      }
      else {
         $status = system ($CFG->params('INIT') . " q");
         if (0 != $status) {
            print_error(213, $INIT_FILE, $!);
            return FAILED;
         }
      }

      if ($srv eq "ohasd") {
         # Start OHASD
         $grep_val = "4640";
	 @output = system_cmd_capture ($crsctl, 'start', 'has');
         $status = shift @output;

         # call $CFG->params('ID')/ohasd install (bug 9133502)
         if (! is_dev_env()) {
            my @cmd = (catfile($CFG->params('ID'), 'ohasd'), 'install');
            system (@cmd);
	 }
      }
      else {
         # Start CRS if it's crsexcl
	 @output = system_cmd_capture ($crsctl, 'start', 'crs', '-excl');
         $status = shift @output;
      }
   } 
   else {
      my $SRVBIN = catfile ($CFG->params('ORACLE_HOME'), "bin", $srv);
      $status = s_run_as_user ("$SRVBIN &", $user);
   }

   if ($grep_val) { @cmdout = grep(/$grep_val/, @output); } # for OHASD

   if (0 == $status) {
      trace ("$srv is starting");
   }
   elsif ($grep_val && scalar(@cmdout) > 0) {
      trace("$srv is already active");
   }
   else {
      print_error(214, $srv);
      trace ("$srv failed to start");
      my $host = tolower_host();
      my $alertlog = catfile ($CFG->params('ORACLE_HOME'), 'log', $host, 
                              "alert${host}.log");

      trace ("Alert log is $alertlog");
      print "Failed to start the Clusterware. Last 20 lines of the alert " .
            "log follow: \n";
      if (-e $alertlog) {
         open(ALERTLOG, "< $alertlog");
         my @lines = reverse <ALERTLOG>;
         my @reversed_lines = reverse(@lines[0..19]);
         my $line;
         foreach $line (@reversed_lines) {
            print $line;
         }
      }
      print "\n";
      return FAILED;
   }

   return SUCCESS;
}

####---------------------------------------------------------
#### Function for checking daemon
# ARGS: 2
# ARG1: daemon to be checked
# ARG2: is daemon running?
sub s_check_service
{
    my ($srv, $isRunning) = @_;
    if (($srv eq "ohasd") && ($isRunning) && (-e $CFG->params('SCRBASE'))) {
        my $AUTOSTARTFILE = catfile($CFG->params('SCRBASE'),
                                    $CFG->HOST, 
                                    $CFG->HAS_USER, "ohasdstr");
        open (AUTOSTART, ">$AUTOSTARTFILE") || die(dieformat(207, $AUTOSTARTFILE, $!));
        print AUTOSTART "enable\n";
        close (AUTOSTART);

        s_set_ownergroup ($CFG->HAS_USER, $CFG->HAS_GROUP, $AUTOSTARTFILE)
	    || die(dieformat(152, $AUTOSTARTFILE));
        s_set_perms ("0644", $AUTOSTARTFILE)
            || die(dieformat(153, $AUTOSTARTFILE));
    }
}

####---------------------------------------------------------
#### Function for initializing SCR settings
# Note: this function will be a no-op on NT
# ARGS: 0
sub s_init_scr
{
    my $crsctl = crs_exec_path('crsctl');
    my $status = system($crsctl, 'create', 'scr', $CFG->params('ORACLE_OWNER'));
    if (0 != $status) {
        print_error(331, catfile($CFG->params('SCRBASE'), $CFG->HOST));
        exit 1;
    }
}

####---------------------------------------------------------
#### Function for running a command as given user
# ARGS: 2
# ARG1: cmd to be executed
# ARG2: user name
sub s_run_as_user
{
    my $user = $_[1];
    my $cmd;

    if ($user) {
        my $SU = "/bin/su";
        $cmd = "$SU $user -c \"$_[0]\"";
        trace ("  Invoking \"$_[0]\" as user \"$user\"");
    } else {
        $cmd = $_[0];
        trace ("  Invoking \"$_[0]\"");
    }

    return system_cmd($cmd);
}

####---------------------------------------------------------
#### Function for running a command as given user, returning back 
#### stdout/stderra output
# ARGS: 3
# ARG1: ref to cmdlist argv list to be executed
# ARG2: user name, can be undef
# ARG3: ref to resulting array of stderr/out, can be undef
sub s_run_as_user2
{
    my $cmdlistref = $_[0];
    my $user = $_[1]; 
    my $capoutref = $_[2];
    my $rc = -1;
    my $SU = "/bin/su";
    
    my @cmdlist;
    if ($user)
    {
      @cmdlist = ( $SU, $user, '-c \'',  @{$cmdlistref}, '\'' );
    }
    else
    {
      @cmdlist = @{$cmdlistref};
    }
    my $cmd = join( ' ', @cmdlist );
    
    # capture stdout/stderr, if requested
    if (defined($capoutref))
    {
      @{$capoutref} = ();
      my $cmdout = tmpnam();

      trace ("s_run_as_user2: Running $cmd");

      # system() with stdout/stderr capture. 
      # Note that this is a portable notation in perl
      # see http://perldoc.perl.org/perlfaq8.html
      # see also
      # http://www.perlmonks.org/?node_id=597613
      open (CMDEXE, "$cmd 2>&1 |" ) 
            or die(dieformat(180, $cmd, $!));
      open (CMDOUT, ">>$cmdout" ) 
            or die(dieformat(208, $cmdout, $!));
      while (<CMDEXE>) { 
	 push( @{$capoutref}, $_ ); 
	 print CMDOUT $_; 
      }
      close (CMDEXE);  # to get $?
      $rc = $?;
      close (CMDOUT);
      s_remove_file ("$cmdout");
    }
    else  # regular system() call
    {
      $rc = s_run_as_user( $cmd, $user );
    }

    if ($rc == 0) {
        trace ("$cmdlist[0] successfully executed\n");
    }
    elsif ($rc == -1) {
        trace ("$cmdlist[0] failed to execute\n");
    }
    elsif ($rc & 127) {
        my $sig = $rc & 127;
        my $core = ($rc & 128) ? 'with' : 'without';
        trace ("$cmdlist[0] died with signal $sig, $core coredump\n");
    }
    else {
        my $retCode = $rc >> 8;
        trace ("$cmdlist[0] exited with rc=$retCode\n");
    }
    return $rc;
}

####---------------------------------------------------------
#### Function for getting value corresponding to a key in ocr.loc or olr.loc
# ARGS: 2
# ARG1: ocr/olr
# ARG2: key
sub s_get_config_key
{
   my $src = $_[0];
   my $key = $_[1];
   $src    =~ tr/a-z/A-Z/;
   my ($val, $cfgfile);

   if ($src eq 'OCR') {
      $cfgfile = $CFG->params('OCRCONFIG');
   }
   elsif ($src eq 'OLR') {
      $cfgfile = $CFG->params('OLRCONFIG');
   }
   elsif ($src eq 'SRV') {
      $cfgfile = $CFG->params('SRVCONFIG');
   }

   # open OCRCONFIG/OLRCONFIG/SRVCONFIG as appropriate
   trace("Opening file $cfgfile");
   open (CFGFL, "<$cfgfile") or return $val;
   while (<CFGFL>) {
      if (/^$key=(\S+)/) {
         $val = $1;
         last;
      }
   }

   close (CFGFL);

   trace("Value ($val) is set for key=$key");
   return $val;
}

####---------------------------------------------------------
#### Function for getting platform family
# ARGS: 0
sub s_get_platform_family
{
    return "unix";
}

####---------------------------------------------------------
#### Function for checking if a path is a symlink, and if so, return the
#### target path
# ARGS: 1
# ARG1: file/dir path
sub s_isLink
{
    my $path = $_[0];
    my $target = "";

    if (-l $path) {
        $target = readlink ($path) or die(dieformat(323, $!));
    }

    return $target;
}

####---------------------------------------------------------
#### Function for getting the old CRS Home
# ARGS:  0
sub s_getOldCrsHome
{
  my ($oldHome, $Name);
  my $CRS_HOME_ENV  = "ORA_CRS_HOME";
  my $INITD 	    = s_getInitd();
  my $OLD_INIT_CSSD = catfile ($INITD, "init.cssd");

  open(FOHM, $OLD_INIT_CSSD) || die(dieformat(324));
  my @buffer = grep(/$CRS_HOME_ENV/, (<FOHM>));
  close FOHM;

  chomp @buffer;
  if (scalar(@buffer) != 0) {
      ($Name, $oldHome) = split(/=/, $buffer[0]);
  }

  return $oldHome;
}

####---------------------------------------------------------
#### Function for stopping the services from OldCrsHome
# ARGS:  1
sub s_stop_OldCrsStack
{
  my $old_crshome = $_[0];
  my @old_version = @{$CFG->oldconfig('ORA_CRS_VERSION')};
  my @output;
  my $status;
   
  if ($old_version[0] eq "10" &&
      $old_version[1] eq "1")
  {
    my $initd 	     = s_getInitd();
    my $old_init_crs = catfile ($initd, "init.crs");
    @output          = system_cmd_capture($old_init_crs, 'stop');
    $status          = shift @output;
    trace ("status=$status");
  }
  else
  {
    my $crsctl = catfile($old_crshome, 'bin', 'crsctl');
    @output    = system_cmd_capture($crsctl, 'stop', 'crs');
    $status    = shift @output;
    trace("status=$status");
  }

  return $status;
}

####---------------------------------------------------------
#### Function for getting the Initd locations
# ARGS:  0
sub s_getInitd
{
  return $INITD;
}

sub s_RemoveInitResources
#-------------------------------------------------------------------------------
# Function: Removing init resources
# Args    : 0
#-------------------------------------------------------------------------------
{
   trace ("Remove init resources");
   my @crs_init_scripts = ('init.evmd', 'init.crsd', 'init.cssd',
			   'init.crs', 'init.ohasd');

   s_remove_itab ("cssd|evmd|crsd|ohasd");
   if (! s_is_Linux_Upstart()) {
      system ($CFG->params('INIT') . " q");
      sleep (5);
   }

   trace ("Removing script for Oracle Cluster Ready services");
   my ($file, $serv);

   foreach $serv (@crs_init_scripts) {
      trace ("Removing " . $CFG->params('ID') . "/$serv file");
      $file = catfile($CFG->params('ID'),$serv);
      s_remove_file ("$file");
   }

   s_clean_rcdirs ("ohasd");
   s_clean_rcdirs ("init.crs");
} 

sub s_ResetOLR
#---------------------------------------------------------------------
# Function: Reset OLR
# Args    : 0
#--------------------------------------------------------------------
{
   trace ("Reset OLR");
   my $bin_dd   = "/bin/dd";
   my $olr_file = s_get_olr_file("olrconfig_loc");

   if (-f $olr_file) {
      trace("Removing OLR file: $olr_file");
      s_remove_file ("$olr_file");
   }
   else {
      trace ("Removing contents from OLR file: $olr_file");
      system ("$bin_dd if=/dev/zero skip=25 bs=4k count=2560 of=$olr_file > $dev_null");
   }

   s_remove_file ($CFG->params('OLRCONFIG'));
}

sub s_ResetOCR
#---------------------------------------------------------------------
# Function: Reset OCR
# Args    : 0
#--------------------------------------------------------------------
{
   trace ("Reset OCR");
   my $bin_dd = "/bin/dd";
   my ($ocr_loc, @mirror_loc, $loc);

   if ($CFG->DOWNGRADE) {
      if ($CFG->oldcrsver eq "9.2") {
         DowngradeTo9i ();
      }
      else {
        DowngradeTo10or11i ();
      }

      return SUCCESS;
   }

   my $olr_file = s_get_olr_file("olrconfig_loc");

   if (-f $olr_file) {
      trace("Removing OLR file: $olr_file");
      s_remove_file ("$olr_file");
   }
   else {
      trace ("Removing contents from OLR file: $olr_file");
      system ("$bin_dd if=/dev/zero skip=25 bs=4k count=2560 of=$olr_file > $dev_null");
   }

   s_remove_file ($CFG->params('OLRCONFIG'));

   if (! $CFG->LASTNODE) {
      s_remove_file ($CFG->params('OCRCONFIG'));
      return SUCCESS;
   }

   if (! -f $CFG->params('OCRCONFIG')) 
   {
      # ocr.loc file does not exist. Take ocr location of srvconfig.loc for setting
      # file permissions
      if (-f $CFG->params('SRVCONFIG')) 
      {
         $ocr_loc =  get_srvdisk ();
      }
   } 
   else {
      $ocr_loc = get_ocrdisk();
      push @mirror_loc, get_ocrmirrordisk();
      push @mirror_loc, get_ocrloc3disk();
      push @mirror_loc, get_ocrloc4disk();
      push @mirror_loc, get_ocrloc5disk();
   }

   my $loc;
   foreach $loc (@mirror_loc) {
      if (($loc) && ($loc ne $dev_null)) {
	 if (-f $loc) {
	    # OCR mirror device is specified and enabled
	    trace("Removing OCR mirror device: $loc");
	    s_remove_file ($loc);
	 }
	 elsif (! isPathonASM($loc)) {
	    trace ("Removing contents from OCR mirror device: $loc");
	    system ("$bin_dd if=/dev/zero skip=25 bs=4k count=2560 of=$loc > $dev_null");
	 }
      }
   }

   # reset OCR device if it's not on ASM
   if (($CFG->LASTNODE) &&
       (! $CFG->DOWNGRADE) &&
       (! $CFG->ASM_STORAGE_USED)) 
   {
      trace ("Removing contents from OCR device");

      if (-f $ocr_loc) {
         trace("Removing OCR device: $ocr_loc");
         s_remove_file ("$ocr_loc");
      }
      elsif (!isPathonASM($ocr_loc)) {
         trace ("Removing contents from OCR device: $ocr_loc");
         system ("$bin_dd if=/dev/zero skip=25 bs=4k count=2560 of=$ocr_loc > $dev_null");
      }
   }

   #remove the ocr.loc in the lastnode in case of ASM storage as well
   if ($CFG->LASTNODE) {
      s_remove_file ($CFG->params('OCRCONFIG'));
   }
   
   # reset permissions of ocr_loc files
   if (-f $ocr_loc) {
      chmod(0644, $ocr_loc);

      if (isOwnerGroupValid()) {
         chown ($CFG->params('ORACLE_OWNER'), $CFG->params('ORA_DBA_GROUP'), 
                $ocr_loc);
      }
   }

   foreach $loc (@mirror_loc) {
      if ((not -z $loc) && (-f $loc)) {
	 chmod(0644, $loc);

	 if (isOwnerGroupValid()) {
            chown ($CFG->params('ORACLE_OWNER'), $CFG->params('ORA_DBA_GROUP'),
                   $loc);
	 }
      }
   }
}

sub s_setParentDirOwner
#-------------------------------------------------------------------------------
# Function: Set $current_dir and its parent directories to $owner/DBA
# Args    : [0] Owner
#           [1] Directory
#-------------------------------------------------------------------------------
{
   my $current_owner = $_[0];
   my $current_dir   = $_[1];
   my $dir           = dirname($current_dir);

   if ((-c $current_dir) || (-b $current_dir))
   {
     trace("The '$dir' is parent directory of a character/block device. "
          ."Skip the action ...");
     return;
   }

   #Save the existing ACLs for parent dir
   trace("saving current owner/permisssios of  parent dir of $dir\n");
   s_saveParentDirinfo($dir);

   while ($dir ne "/" && $dir ne ".") {
      s_set_ownergroup($current_owner,
                       $CFG->params('ORA_DBA_GROUP'),
                       $dir) || die(dieformat(152, $dir));
      s_set_perms ("0755", $dir);
      $dir = dirname($dir);
   }
}

sub s_resetParentDirOwner
#-------------------------------------------------------------------------------
# Function: Set $current_dir and its parent directories to $owner/DBA
# Args    : [0] Directory
#-------------------------------------------------------------------------------
{
   my $current_dir = $_[0];
   my $dir = dirname($current_dir);
   my $savepermfile = catfile ($CFG->ORA_CRS_HOME, "crs", "install", 
			       "ParentDirPerm_" . $CFG->HOST . ".txt");

   if (-e $savepermfile) {
      trace("reset ACLs  of parent dir of grid home\n");
      my ($fname, $fuser, $fgroup, $fperm);
      open (PFILE, $savepermfile) || print_error(215, $savepermfile, $!);
      while(<PFILE>) {
	 ($fname, $fuser, $fgroup, $fperm) = split(/:/, $_);
         trace("Got $fname:$fuser:$fgroup:$fperm from $savepermfile");
	 s_set_ownergroup ($fuser, $fgroup, $fname);
	 s_reset_perms ($fperm, $fname);
      }

      close (PFILE);

      s_remove_file($savepermfile);
   }
}

####---------------------------------------------------------
#### Function for resetting permissions on a specified path
#### to the input permissions value.
# ARGS : 2
# ARG1 : permissions
# ARG3 : file/dir
sub s_reset_perms
{
    my ($perms, $file) = @_;

    if (!$perms) {
        print_error(43);
        return FAILED;
    }

    if (!$file) {
        print_error(42);
        return FAILED;
    }

    if (!(-e $file)) {
        print_error(46, $file);
        return "FAILED";
    }

   if ($CFG->DEBUG) { trace ("Setting permissions ($perms) on file/dir $file"); }

   if (! chmod (oct($perms), $file)) {
      print_error(153, $file);
      trace("Can't change permissions of $file: $!");
      return FAILED;
   }

   return SUCCESS;
}

sub s_ResetVotedisks
#-------------------------------------------------------------------------------
# Function: Reset voting disks
# Args    : [0] list of voting disks
#-------------------------------------------------------------------------------
{
   trace ("Reset voting disks");
   my @votedisk_list = @_;
   foreach my $vdisk (@votedisk_list) {
     if (-f $vdisk) {
       trace("Removing voting disk: $vdisk"); 
       s_remove_file ("$vdisk");
     }
     elsif (! isOCRonASM()) { 
       trace ("Removing contents from voting disk: $vdisk");
       my $bin_dd = "/bin/dd";
       system ("$bin_dd if=/dev/zero skip=25 bs=4k count=2560 of=$vdisk > $dev_null");
     }
   }
}

sub s_CleanTempFiles
#-------------------------------------------------------------------------------
# Function: Remove misc files and directories
# Args    : none
#-------------------------------------------------------------------------------
{
   my ($dir, $file);

   # call $CFG->params('ID')/ohasd deinstall (bug 9133502)
   if (! is_dev_env()) {
      my @cmd = (catfile($CFG->params('ID'), 'ohasd'), 'deinstall');
      system (@cmd);
   }

   # remove /etc/init.d/ohasd
   my $initd = s_getInitd();
   $file  = (catfile ($initd, "ohasd"));
   s_remove_file ("$file");

   # remove /var/tmp/.oracle
   $dir = catdir ("/var", "tmp", ".oracle");
   if (-e $dir) {
      trace ("Remove $dir");
      rmtree ($dir);
   }

   # remove /ect/inittab.crs
   $file = catfile ("/etc", "inittab.crs");
   s_remove_file ("$file");

   # remove /tmp/.oracle
   $dir = catdir ("/tmp", ".oracle");
   if (-e $dir) {
      trace ("Remove $dir");
      rmtree ($dir);
   }

   # remove /ect/oracle/lastgasp
   if ($CFG->defined_param('OLASTGASPDIR')) {
      $dir = $CFG->params('OLASTGASPDIR');
      if (-e $dir) {
         trace ("Remove $dir");
         rmtree ($dir);
      }
   }

   # remove /ect/oratab
   if (! $CFG->SIHA) {
      if ($CFG->defined_param('HOME_TYPE')) {
         $file = catfile ("/etc", "oratab");
         s_remove_file ("$file");
      }
   }

   # remove /etc/oracle if empty
   if (-e $CFG->params('OCRCONFIGDIR')) {
      # remove backup ocr.loc.orig and olr.loc.orig files
      $file  = ($CFG->params('OCRCONFIG') . ".orig");
      s_remove_file ("$file");
      $file  = ($CFG->params('OLRCONFIG') . ".orig");
      s_remove_file ("$file");

      # check if it's empty
      opendir (DIR, $CFG->params('OCRCONFIGDIR'));
      my @files = readdir(DIR);
      close DIR;

      if (scalar(@files) == 2) {
         trace ("Remove " . $CFG->params('OCRCONFIGDIR'));
         rmtree $CFG->params('OCRCONFIGDIR');
      }
   }
}

sub s_checkOracleCM
#-------------------------------------------------------------------------------
# Function: Check for OracleCM by checking libskgxn on unix.
# Args    : none
# Return  : TRUE - if found
#-------------------------------------------------------------------------------
{
   my $false = 0;
   my $true  = 1;
   my $libskgxnBase_lib = catfile('/etc', 'ORCLcluster', 'oracm', 'lib',
                                  'libskgxn2.so');
   my $libskgxn_lib     = catfile('/opt', 'ORCLcluster', 'lib', 'libskgxn2.so');
   trace("libskgxnBase_lib = $libskgxnBase_lib");
   trace("libskgxn_lib = $libskgxn_lib");

   if ((-e $libskgxn_lib) || (-e $libskgxnBase_lib)) {
      # no SKGXN;
      trace("SKGXN library file exists");
      return $true;
   }

   trace("SKGXN library file does not exists");
   return $false;
}

sub s_createConfigEnvFile
#-------------------------------------------------------------------------------
# Function: Create s_crsconfig_$HOST_env.txt file for Time Zone
# Args    : none
# Notes   : Valid <env_file> format
#           (Please keep this in sync with has/utl/crswrapexec.pl)
#             * Empty lines: lines with all white space
#             * Comments: line starts with #.
#             * <key>=<value>
#             * <key> is all non-whitespace characters on the left of the
#               first "=" character.
#             * <value> is everything on the right of the first "=" character
#               (including whitespaces).
#             * Surrounding double-quote (") won't be stripped.
#             * Key with blank <value> ('') will be undefined.
#               (e.g: Hello=, Hello will be undefined)
#-------------------------------------------------------------------------------
{
   my $env_file = catfile($CFG->ORA_CRS_HOME, 'crs', 'install',
                          's_crsconfig_' . $CFG->HOST . '_env.txt');

   open (ENVFILE, ">$env_file") or die(dieformat(255, $env_file, $!));

   print ENVFILE "### This file can be used to modify the NLS_LANG environment"
               . " variable, which determines the charset to be used for messages.\n"
               . "### For example, a new charset can be configured by setting"
               . " NLS_LANG=JAPANESE_JAPAN.UTF8 \n"
               . "### Do not modify this file except to change NLS_LANG,"
               . " or under the direction of Oracle Support Services\n\n";

   # get TZ
   if ($CFG->defined_param('TZ')) {
      my $tz = $CFG->params('TZ');
      $tz    =~ s/'//g; # remove single quotes
      print ENVFILE "TZ=" . $tz . "\n";
   }

   # get NLS_LANG
   if ($CFG->defined_param('LANGUAGE_ID')) {
      my $nls_lang = $CFG->params('LANGUAGE_ID');
      $nls_lang =~ s/'//g; # remove single quotes
      print ENVFILE "NLS_LANG=" . $nls_lang . "\n";
   }

   # Set RT_GRQ to force real-time processes onto global run-queue for AIX
   # NOTE - ONLY FOR AIX, DOES NOT MAKE SENSE FOR OTHER PLATFORMS
   #        See IBM AIX documentation on RT_GRQ parameter for more information
   if ($OSNAME eq 'aix') {
     print ENVFILE "RT_GRQ=ON\n";
     print ENVFILE "EXTSHM=OFF\n";
   }

   # Make sure that env var TNS_ADMIN will be unset.
   print ENVFILE "TNS_ADMIN=\n";

   # Bug 13506877: Do not unset ORACLE_BASE; CLSB and ADR need it now.
#  print ENVFILE "ORACLE_BASE=\n";

   close (ENVFILE);

   s_set_ownergroup ($CFG->SUPERUSER,
                     $CFG->params('ORA_DBA_GROUP'), 
                     $env_file) || die(dieformat(152, $env_file));
   s_set_perms ("0750", $env_file) || die(dieformat(153, $env_file));
}

sub s_isRAC_appropriate
#-------------------------------------------------------------------------------
# Function:  Check if rac_on/rac_off on Unix
# Args    :  none
# Returns :  TRUE  if rac_on/rac_off     needs to be set
#            FALSE if rac_on/rac_off not needs to be set
#-------------------------------------------------------------------------------
{
   my $rdbms_lib = catfile($CFG->ORA_CRS_HOME, "rdbms", "lib");
   my $success   = TRUE;

   # save current dir
   my $save_dir = getcwd;

   # check for rac_on
   chdir $rdbms_lib;

   my $cmd = "$ARCHIVE -tv libknlopt.a | grep kcsm";

   open ON, "$cmd |";
   my @kcsm = (<ON>);
   close ON;

   if (scalar(@kcsm) == 0) {
      if (! $CFG->SIHA) {
         $success = FALSE;
         print " \n";
         print "The oracle binary is currently linked with RAC disabled.\n";
         print "Please execute the following steps to relink oracle binary\n";
         print "as the oracle user and rerun the command with RAC enabled: \n";
         print "   setenv ORACLE_HOME <crshome> \n";
         print "   cd <crshome>/rdbms/lib \n";
         print "   make -f ins_rdbms.mk rac_on ioracle \n";
      }
   }
   elsif ($CFG->SIHA) {
      $success = FALSE;
      print " \n";
      print "The oracle binary is currently linked with RAC enabled.\n";
      print "Please execute the following steps to relink oracle binary\n";
      print "as the oracle user and rerun the command with RAC disabled: \n";
      print "   setenv ORACLE_HOME <oracle_restart_home> \n";
      print "   cd <oracle_restart_home>/rdbms/lib \n";
      print "   make -f ins_rdbms.mk rac_off ioracle \n";
   }

   # restore save_dir
   chdir $save_dir;

   return $success;
}

sub s_configureCvuRpm
#------------------------------------------------------------------------------
# Function:  Install cvuqdisk rpm on Linux
# Args    :  none
#-------------------------------------------------------------------------------
{
    my $uname =`uname`;
    chomp($uname);

    if ($uname=~/Linux/)
    {
       trace ("Install cvuqdisk rpm on Linux...");
       my $rmpexe = "/bin/rpm";
       my $install_cvuqdisk=FALSE;
       my $rpm_pkg_dir;
       my $rpm_file;

       if (is_dev_env())
       {
          $rpm_pkg_dir=catfile($CFG->ORA_CRS_HOME,'opsm','cv','remenv');
       }
       else
       {
          $rpm_pkg_dir=catfile($CFG->ORA_CRS_HOME,'cv', 'rpm');
       }

       opendir (DIR, $rpm_pkg_dir);
       foreach (sort grep(/cvuqdisk/,readdir(DIR)))
       {
           $rpm_file = $_;
       }
       closedir DIR;

       my $new_rpm_file = $rpm_pkg_dir . "/" . $rpm_file;
       trace ("New package to install is $new_rpm_file");

       trace ("Invoking \"$rmpexe -q cvuqdisk\" command");
       my $curr_rpm_version = `$rmpexe -q cvuqdisk --queryformat '%{VERSION}'`;
       my $status = $?;

       if ($status == 0) 
       {
          my $curr_rpm_release = `$rmpexe -q cvuqdisk --queryformat '%{RELEASE}'`;

          trace ("Invoking \"$rmpexe -qp $new_rpm_file\" command");
          my $new_rpm_version = `$rmpexe -qp $new_rpm_file --queryformat '%{VERSION}'`;
          my $new_rpm_release = `$rmpexe -qp $new_rpm_file --queryformat '%{RELEASE}'`;

          chomp ($curr_rpm_version);
          chomp ($new_rpm_version);
          chomp ($curr_rpm_release);
          chomp ($new_rpm_release);

          if ($curr_rpm_version eq "package cvuqdisk is not installed")
          {
              trace ("package is not installed");
              $install_cvuqdisk = TRUE;
          }
          else
          {
              trace ("check package versions new = [$new_rpm_version];old=[$curr_rpm_version] ");
              my $i;
              my @currPkgArr = split(/\./, $curr_rpm_version);
              my @newPkgArr = split(/\./, $new_rpm_version);
              my $loopMax = (scalar @currPkgArr <= scalar @newPkgArr)?scalar @currPkgArr:scalar @newPkgArr;
              my $currentConfigGood = FALSE;

              for ($i=0;$i<$loopMax;$i++) {
                 if ($currPkgArr[$i] == $newPkgArr[$i])
                 {
                    next;
                 }
                 if ($currPkgArr[$i]  > $newPkgArr[$i])
                 {
                    $currentConfigGood = TRUE;
                    last;
                 }
                 trace ("install new package for version");
                 $install_cvuqdisk = TRUE;
                 last;
              }

              if (!$currentConfigGood && !$install_cvuqdisk && ($curr_rpm_release lt $new_rpm_release))
              {
                  trace ("install new package for release");
                  $install_cvuqdisk = TRUE;
              }
          }
       }
       else
       {
           trace ("no existing cvuqdisk found");
           $install_cvuqdisk = TRUE;
       }

       if ($install_cvuqdisk)
       {
          my $orauser  = $CFG->params('ORACLE_OWNER');
          my $CVUQDISK_GRP=`id -gn $orauser`;
          chomp($CVUQDISK_GRP);
          $ENV{'CVUQDISK_GRP'} = $CVUQDISK_GRP;

          trace ("removing old rpm");
          my @args = ($rmpexe, "-e", "--allmatches", "cvuqdisk");
          my @out = system_cmd_capture (@args);
          my $rc  = shift @out;

          if ( $rc == 1)
          {
              trace ("Older version cvuqdisk not uninstalled");
          }

          trace ("installing/upgrading new rpm");
          system ("$rmpexe -Uv $new_rpm_file");
       }
    }
    return TRUE;
}

sub s_removeCvuRpm
#---------------------------------------------------------------------
# Function: Remove cvuqdisk rpm
# Args    : None
#---------------------------------------------------------------------
{
    my $uname =`uname`;
    chomp($uname);

    if ($uname=~/Linux/)
    {
       trace ("removing cvuqdisk rpm");
       system ("/bin/rpm -e --allmatches cvuqdisk");
    }
    return TRUE;
}

sub s_createLocalOnlyOCR
#-------------------------------------------------------------------------------
# Function:  Create local-only OCR
# Args    :  none
#-------------------------------------------------------------------------------
{
   trace ("create Local Only OCR on Linux...");

   my $owner      = $CFG->params('ORACLE_OWNER');
   my $dba_group  = $CFG->params('ORA_DBA_GROUP');
   my $ocr_config = $CFG->params('OCRCONFIG');

   # create ocr.loc w/ local_only=TRUE and set ownergroup
   open (FILEHDL, ">$ocr_config") or die(dieformat(255, $ocr_config, $!));
   print FILEHDL "local_only=TRUE\n";
   close (FILEHDL);
   s_set_ownergroup ($owner, $dba_group, $ocr_config)
                                     || die(dieformat(152, $ocr_config));
   s_set_perms ("0640", $ocr_config) || die(dieformat(153, $ocr_config));
}

sub s_houseCleaning
#-------------------------------------------------------------------------------
# Function:  Remove entries from inittab and misc files
# Args    :  none
#-------------------------------------------------------------------------------
{
   # remove cssd/evmd/crsd entries from inittab
   s_remove_itab ("cssd|evmd|crsd");

   # remove /etc/init.d/init.crs
   my $file = catfile ($CFG->params("ID"), 'init.crs');
   if (-f $file) {
      trace ("remove $file");
      s_remove_file ("$file");
   }

   # remove /etc/init.d/init.crsd
   $file = catfile ($CFG->params("ID"), 'init.crsd');
   if (-f $file) {
      trace ("remove $file");
      s_remove_file ("$file");
   }

   # remove /etc/init.d/init.cssd
   $file = catfile ($CFG->params("ID"), 'init.cssd');
   if (-f $file) {
      trace ("remove $file");
      s_remove_file ("$file");
   }

   # remove /etc/init.d/init.evmd
   $file = catfile ($CFG->params("ID"), 'init.evmd');
   if (-f $file) {
      trace ("remove $file");
      s_remove_file ("$file");
   }

   # remove S96init.crs files
   my $search_dir = catdir ('/etc', 'rc.d');
   my $delete_file = "s96init.crs";

   finddepth (\&remove_file, $search_dir);

   # remove K96init.crs files
   $delete_file = "k96init.crs";

   finddepth (\&remove_file, $search_dir);

   # remove K19init.crs files
   $delete_file = "k19init.crs";

   finddepth (\&remove_file, $search_dir);

   sub remove_file {
      if (/\b$delete_file\b/i) {
         trace ("remove $File::Find::name");
         s_remove_file ("$_");
      }
   }
}

sub s_is92ConfigExists
#-------------------------------------------------------------------------------
# Function: Check if config exists in 9.2
# Args    : none
# Returns : TRUE  if     exists
# 	    FALSE if not exists
#-------------------------------------------------------------------------------
{
   my $srvconfig_loc;

   trace("SRVCONFIG=" . $CFG->params('SRVCONFIG'));

   if (-f $CFG->params('SRVCONFIG')) {
      trace ("Checking repository used for 9i installations");
      $srvconfig_loc = s_get_config_key ("srv", "srvconfig_loc");

      trace("srvconfig location=<$srvconfig_loc>");

      if ($srvconfig_loc eq '/dev/null') {
          trace("srvconfig location=<$srvconfig_loc>");
          trace("Oracle 92 configuration and SKGXN library does exists");
          return TRUE;
      }
   }

   trace("Oracle 92 configuration and SKGXN library does not exists");

   return FALSE;
}

sub s_copyOCRLoc
#-------------------------------------------------------------------------------
# NOTES: OCR handles the ocr.loc updates on its own, and this is kept around 
#	 as a fallback
#-------------------------------------------------------------------------------
{
   my $cluutil 	   = catfile ($CFG->ORA_CRS_HOME, 'bin', 'cluutil');
   my $ocrloc_temp = catfile ($CFG->ORA_CRS_HOME, 'srvm', 'admin', 'ocrloc.tmp');
   my $ocrloc_file = catfile ($CFG->ORA_CRS_HOME, 'srvm', 'admin', 
                              $CFG->params('OCRLOC'));
   my @node_list   = getCurrentNodenameList();
   my $success	   = FALSE;
   my @capout	   = ();
   my @cmd;
   my $rc;

   if (! (-e $cluutil)) {
      trace("$cluutil not found");
      trace("Unable to copy OCR locations");
      return FALSE;
   }

   my $host = $CFG->HOST;
   foreach my $node (@node_list) {
      if ($node !~ /\b$host\b/i) {
         @cmd = ("$cluutil", '-sourcefile', $CFG->params('OCRCONFIG'), 
                 '-sourcenode', $node, '-destfile', $ocrloc_temp, 
                 '-nodelist', $node);
         $rc = run_as_user2($CFG->params('ORACLE_OWNER'), \@capout, @cmd);

         if ($rc == 0) {
            trace("@cmd ... passed");
	    $success = TRUE;
	    last;
         }
         else {
            trace("@cmd ... failed");
	    if (scalar(@capout) > 0) {
	    trace("capout=@capout");
 	    }
         }
      }
      else {
         trace("Avoiding self copy of ocr.loc on node: $node");
      }
   }

   if ($success) {
      rename ($ocrloc_temp, $ocrloc_file);
   } else {
      trace("@cmd ... failed");
      s_remove_file ("$ocrloc_temp");
   }

   return TRUE;
}

sub s_removeGPnPprofile
#-------------------------------------------------------------------------------
# Function: Remove all contents under $crshome/gpnp dir
# Args    : none
#-------------------------------------------------------------------------------
{
   my $dir = catdir($CFG->ORA_CRS_HOME, 'gpnp');

   # read dir contents
   opendir (DIR, $dir);
   my @files = readdir(DIR);
   close DIR;

   my $file;
   foreach $file (@files) {
      if ($file eq '.' || $file eq '..') {
         next;
      }
      elsif (-f "$dir/$file") {
         trace ("remove file=$dir/$file");
         s_remove_file ("$dir/$file");
      }
      elsif (-d "$dir/$file") {
         trace ("rmtree dir=$dir/$file");
         rmtree ("$dir/$file");
      }
   }
}

sub s_remove_file
{
   my $remfile = $_[0];

   if (-e $remfile || -l $remfile) {
      my @args = ("rm", $remfile);

      trace("Removing file $remfile");
      my @out = s_system_cmd_capture2(@args);
      my $rc  = shift @out;

      if ($rc == 0) {
         trace("Successfully removed file: $remfile");
      }
      else {
         trace("Failed to remove file: $remfile");
         return FAILED;
      }
   }

   return SUCCESS;
}

sub s_system_cmd_capture2 {

  my $rc = 0;
  my @output;

  @output = `@_ 2>&1`;
  $rc = $? >> 8;

  if (($rc != 1) && ($rc & 127)) {
    # program returned error code
    my $sig = $rc & 127;
    trace("Failure with return code $sig from command: @_");
  }
  elsif ($rc) { 
    trace("Failure with return code $rc from command @_"); 
  }

  if ($CFG->DEBUG) { trace("@output"); }

  return ($rc, @output);
}

sub s_getAbsLink
#-------------------------------------------------------------------------------
# Function: Get absolute link
# Args    : file_loc
# Returns : link_loc
#-------------------------------------------------------------------------------
{
   my $file_loc = $_[0];
   my $link_loc = readlink($file_loc);
   my $count    = 0;

   $count++ while ($link_loc =~ /\.\./g);

   if ($count > 0) {
      trace("symbolic link is not absolute link: $link_loc");
      my @ocr_dir       = split(/\//, $file_loc);
      my @link_dir      = split(/\.\./, $link_loc);
      my $ocr_dir_cntr  = scalar @ocr_dir;
      my $abs_link      = '/';
      my $limit         = $ocr_dir_cntr - $count - 1;
      my $i;

      for ($i = 0; $i < $limit; $i++) {
          $abs_link = catdir($abs_link, $ocr_dir[$i]);
      }

      $link_loc = catfile($abs_link, $link_dir[$count]);
      trace("absolute link is $link_loc");
   }

   return $link_loc;
}

sub s_crf_check_bdbloc
{
  my $bdbloc = $_[0];

  trace("CHM/OS repository location checks: $bdbloc");

  # check for existence first
  if (! -d $bdbloc)
  {
    trace("INFO: CHM/OS repository path $bdbloc does not exist. Creating...\n");
    mkpath($bdbloc);
  }

  if (! -w $bdbloc)
  {
    trace("INFO: CHM/OS repository path $bdbloc is not writable, changing ");
    trace("permissions on it...\n");
    s_set_perms ("0755", $bdbloc);
  }

  # check for space now. df -k reports 1K blocks on linux/solaris.
  # Check for 2GB per node.
  my $rqrd;
  my $nodelist = $_[1];
  my $df;

  # For storing the df -k/bdf index as it is same on Linux, Solaris, 
  # HPUX but different on AIX. In AIX it is 3rd column for available 
  # space where as in others it is 4
  my $avail_index;
  
  if ($^O =~ /aix/i)
  {
    $avail_index = 2;
  }
  else
  {
    $avail_index = 3;
  } 
  
  # Linux and Solaris has df command but HPUX doesn't have it.
  # For fixing Bug 11675864. HPUX uses bdf comand.
  
  if ($^O =~ /hpux/i)
  {
    $df =  "/usr/bin/bdf";
  }
  else
  {
    $df = "/bin/df -k";
  }

  chomp($nodelist);
  my @hosts = split(/[,]+/, $nodelist);
  my $tlang = $ENV{LANG};
  $ENV{LANG} = "";

  $rqrd = 1024*1024;

  if (open(DFH, "$df $bdbloc 2>/dev/null |"))
  {
    $ENV{LANG} = $tlang;
    my $fulline="";
    while (<DFH>)
    {
      chomp();
      if (!($_ =~ m/Filesystem.*/i))
      {
        $fulline .= $_;
      }
    }
    close DFH;
    my @parts = split(/[ \n]+/, $fulline);
    my $avl = $parts[$avail_index];
    if ($avl < $rqrd)
    {
      print_error(216, $bdbloc);
      trace ("Insufficient free space available in CHM/OS repository $bdbloc;" .
            "required space is $rqrd KB and available space is $avl KB;" .
            " run 'oclumon manage -repos reploc <new_path>' command to change" .
            " CHM/OS"); 
      return 12; #correct error code for out of space.
    }
    $rqrd += $rqrd/3;
    if ($avl < $rqrd)
    {
      trace("WARNING: File system hosting the CHM/OS repository $bdbloc is");
      trace(" running low on disk space. Available disk space=$avl KB. ");
    }
  }
  $ENV{LANG} = $tlang;
}

sub s_crf_remove_itab
{
  trace("Removing /etc/init.d/init.crfd");
  unlink("/etc/init.d/init.crfd");

  # cleanup from the auto startup configuration
  s_remove_itab ("crfd");

  if ( ! s_is_Linux_Upstart()) {
     system ($CFG->params('INIT') . " q");
  }
}

sub s_is_primeCluster
{
    my $status;
    my $checksc = "/usr/bin/cftool";
    if (! (-e $checksc)) {
      trace ("$checksc not found");
      return FALSE;
    }
    $status = system("$checksc -l");
    if ($status == 0) {
        return TRUE;
    } else {
        return FALSE;
    }
}

sub s_is_sunCluster
{
    my $status;
    my $checksc = "/usr/sbin/clinfo";
    if (! (-e $checksc)) {
      trace ("$checksc not found");
      return FALSE;
    }
    $status = system("$checksc");
    if ($status == 0) {
        return TRUE;
    } else {
        return FALSE;
    }
}

# we cannot support solaris 5.11 ipmp interfaces
sub s_is_sun_ipmp
{
  my $ifconfig = "/usr/sbin/ifconfig";
  my ($net, $inf, $sub, $type, $tmp, $rc, @out);

  # split apart the network keys and search
  foreach $net (split(/,/, $CFG->params('NETWORKS')))
  {
    ($inf, $tmp)  = split(/\//, $net);
    ($sub, $type) = split(/:/, $tmp);
    
    # we only care about cluster_interconnect keys
    next if( $type !~ /cluster_interconnect/ );

    # lets call ifconfig to figure out information about this interface
    ($rc, @out) = system_cmd_capture($ifconfig, $inf);

    # skip any interfaces on which we had errors
    next if( $rc != 0 );

    # if we find any cluster_interconnect keys that have IPMP flag set
    # on their interface description, then we are done
    if( $out[0] =~ /IPMP/ )
    {
      return TRUE;
    }
  }

  # made it all the way out without finding any IPMP private
  return FALSE;
}

# we only create the HAIP resources for CRS installs and for now only on
# linux, hpux, solaris, aix but eventually this should be for all unix.
# NO HAIP if this is sun cluster or solaris 5.11 with IPMP as private
# NO HAIP if env-var HAIP_UNSUPPORTED is set for all platforms
sub s_is_HAIP_supported
{
  if((($OSNAME eq "linux") || ($OSNAME eq "hpux") || ($OSNAME eq "aix") ||
     (($OSNAME eq "solaris") && (!s_is_sunCluster()) && (!s_is_primeCluster())
      && (!s_is_sun_ipmp()))) && (!$ENV{'HAIP_UNSUPPORTED'}))
  {
    return TRUE;
  }
  else {
    return FALSE;
  }
}

# On some platforms we will install the HAIP resource, but we won't worry
# about it's failure to come up but continue on regardless
sub s_is_HAIP_NonFatal
{
  if( ($OSNAME eq "hpux") || ($OSNAME eq "aix") )
  {
    return TRUE;
  }
  else 
  {
    return FALSE;
  }
}

# In Linux, there are two startup mechanisms. SysVInit and the newer upstart.
# This routine will check if upstart is being used or not.
# returns TRUE if upstart is being used for the init
sub s_is_Linux_Upstart
{
  # Check not cached, check if it is upstart
  if ( $UPSTART_USED == -1 ) {
     if ($OSNAME eq 'linux')
     {
        my $rpm = '/bin/rpm';
        my @cmd = ($rpm, '-qf',  '/sbin/init');
        my ($rc, @grepout, @cmdout);
  
       # Check if rpm is present
       if (! (-e $rpm)) {
         print_error(46, $rpm);
         return FALSE;
       }

       # Run the rpm command
       ($rc, @cmdout) = system_cmd_capture(@cmd);
       if ( $rc != 0) { return FALSE; }
    
       # check for upstart in rpm output
       @grepout = grep(/^UPSTART-/i, @cmdout);
       if (scalar(@grepout) > 0) { $UPSTART_USED=TRUE; return TRUE; }
     }
     # Non-Linux
     $UPSTART_USED=FALSE;
     return FALSE;
  }
  # return the cached value
  return $UPSTART_USED;
}

# Add the oracle-ohasd conf file from CRSHOME to /etc/init
sub s_add_upstart_conf
{
   my ($status, @output);
   my $UPSTART_INIT_DIR = $CFG->params('UPSTART_INIT_DIR');
   my $UPSTART_OHASD_CONF_CH = catfile($CFG->ORA_CRS_HOME, 'crs', 'install',
                                       $UPSTART_OHASD_CONF_FILE);
   my $UPSTART_OHASD_CONF    = catfile ($UPSTART_INIT_DIR, $UPSTART_OHASD_CONF_FILE);

   if ( -e $UPSTART_OHASD_CONF_CH ) {
       # Copy $UPSTART_OHASD_CONF_CH to /etc/init/
       unless (copy ("$UPSTART_OHASD_CONF_CH", "$UPSTART_OHASD_CONF")) {
       print_error(105, "$UPSTART_OHASD_CONF_CH", "$UPSTART_OHASD_CONF", $!);
       return FAILED;
       }
   }
   else {
       print_error(46, $UPSTART_OHASD_CONF_CH);
       return FAILED;
   }
   
   # Restore the default SELinux security context
   if ( -e $RESTORECON ) {
      ($status, @output) = system_cmd_capture ("$RESTORECON -iF $UPSTART_OHASD_CONF");
   }
   return SUCCESS;
}

#### Function for removing the Oracle conf files from /etc/init
#  ARGS : 1 - match pattern for the Oracle conf file pattern
#  NOTE : Oracle conf files will be of the format : oracle-{name}.conf like oracle-ohasd.conf
sub s_remove_upstart_conf
{
   my $INITCTL = $CFG->params('INITCTL');
   my $UPSTART_INIT_DIR = $CFG->params('UPSTART_INIT_DIR');
   my ($status, @output, @initctl_list, @file_list, $file, $service, $service_name);
   my $match_pattern = $_[0];

   # 
   # Construct service and then stop it
   #

   # Get list of registered services in the system
   ($status, @output) = system_cmd_capture ("$INITCTL list");
   if(0 != $status)
   {
     # Failed to list INITCTL
     print_error(218, $!);
     return FAILED;
   }

   # Filter the pattern of services interested from the initctl list
   @initctl_list = grep(/$match_pattern/i, @output);
   
   # Stop any running services from the list
   foreach $service(@initctl_list)
   {
       # Check if it is in running state and stop it
       if (scalar(grep(/running/,$service) > 0))
       {
           ($service_name)  = split(/ /, $service);
           trace("Service [$service_name] running.\n");
           ($status, @output) = system_cmd_capture ("$INITCTL stop $service_name");
           if(0 != $status)
           {
             # Failed to stop service 
             print_error(218, $!);
             return FAILED;
           }
       }
   }

   # 
   # Remove the corresponding conf files  
   #  

   # Construct the file path pattern for the upstart conf files
   #
   for ($match_pattern) {
      # Replace the | with space of the match_pattern
      s-\|- -g;
      # Convert pattern "ohasd cssd" to "/etc/init/oracle-*ohasd*.conf /etc/init/oracle-*cssd*.conf "
      s-^-${UPSTART_INIT_DIR}/oracle\-*-g;
      s- - ${UPSTART_INIT_DIR}/oracle\-*-g;
      s- -*.conf -g;
      s-$-*.conf -g;
   }

   # Get the file list for the pattern 
   (@file_list) = glob ($match_pattern);
   trace ("Glob file list = @file_list");

   # Remove the conf file
   foreach $file(@file_list) {
     s_remove_file("$file");
   }

   return SUCCESS;
}

sub s_getfileinfo
{
  my $file = $_[0];
  my ($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,$atime,$mtime,$ctime,
      $blksize,$blocks);
  my $user;
  my $group;
  my $permissions;

  trace("Getting file permissions for $file\n");
  if (($dev,$ino,$mode,$nlink,$uid,$gid,$rdev,$size,$atime,$mtime,$ctime,
       $blksize,$blocks) = lstat($file))
  {
        $user = getpwuid($uid);
        $group = getgrgid($gid);
        $permissions = sprintf "%04o", S_IMODE($mode);

        return($file, $user , $group, $permissions);
  }
}

sub s_saveParentDirinfo
#-------------------------------------------------------------------------------
# Function: save $current_dir and its parent directories ACL
# Args    : [0] Directory
#-------------------------------------------------------------------------------
{
   my $dir = $_[0];
   my $savepermfile = catfile ($CFG->ORA_CRS_HOME, "crs", "install",
                               "ParentDirPerm_" . $CFG->HOST . ".txt");
   my ($fname, $fuser, $fgroup, $fperm, @perm_table);
   trace("saving parent dir ACLs in $savepermfile\n");

   if (-e $savepermfile) {
      # load into @perm_table
      trace("loading... $savepermfile\n");
      @perm_table = read_file ($savepermfile);
   }

   my $save_entries = scalar(@perm_table);
   while ($dir ne "/" && $dir ne ".") {
      ($fname, $fuser, $fgroup, $fperm) = s_getfileinfo($dir);
      my $text = "$fname:$fuser:$fgroup:$fperm";
      my $search_str = $fname . ':';

      # check if it exists
      my $found = FALSE;
      foreach my $rec (@perm_table) {
         chomp($rec);
         if ($rec =~ m/^$search_str/) {
            $found = TRUE;
          }
      }

      if (! $found) {
         push @perm_table, $text;
      }

      $dir = dirname($dir);
   }

   # if new record is added to the @perm_table, create new perm file
   if (scalar(@perm_table) != $save_entries) {
      if (s_remove_file($savepermfile)) {
         open (SFILE, ">$savepermfile") or
            print_error(217, $savepermfile, $!);

         foreach my $dir (@perm_table) {
            trace("writing $dir to $savepermfile");
            print SFILE "$dir\n";
         }

         close SFILE;
      }
   }

   s_set_ownergroup ($CFG->SUPERUSER, $CFG->params('ORA_DBA_GROUP'), 
                     $savepermfile);
   s_set_perms ("0644", $savepermfile) ;
}

sub s_install_initd
{
   #Would skip this install in dev environment
   if (s_isSUSELinux() && ! is_dev_env()) {
      my $ohasd  = catfile($CFG->params('ID'), 'ohasd');
      my $initd  = catfile('/usr', 'lib', 'lsb', 'install_initd');
      my @out    = system_cmd_capture($initd, $ohasd);
      my $status = shift @out;

      if ($status != 0) {
         print_error(218, $!);
         return FAILED;
      }
   }

   return SUCCESS;
}

sub s_remove_initd
{
   # remove initd if it's SUSE Linux
   if (s_isSUSELinux()) {
      my $ohasd  = catfile($CFG->params('ID'), 'ohasd');
      my $initd  = catfile('/usr', 'lib', 'lsb', 'remove_initd');
      my @out    = system_cmd_capture($initd, $ohasd);
      my $status = shift @out;

      if ($status != 0) {
         trace ("Failed to remove initd");
      }
   }
}

sub s_isSUSELinux
{
   if ($OSNAME eq 'linux')
   {
      my @cmd = (catfile('/bin', 'rpm'), '-q', 'sles-release');
      my @out = system_cmd_capture(@cmd);
      my $rc  =  shift @out;
      if ($rc == 0) {
         return TRUE;
      }
      else {
         return FALSE;
      }
   }else{
     return FALSE;
   }
}

#------------------------------------------------------------------------------
# Function: check if network configuration files need to be updated to 
#           disable 169.254/16 route usage (ZEROCONF)
#------------------------------------------------------------------------------
sub s_CheckNetworkConfig
{
  my $fixed = TRUE;
  my $base;
  my $temp;
  my $save;

  # network configuration modifications only valid for linux
  if( !($OSNAME eq "linux") )
  {
    return;
  }

  if( is_dev_env() ) 
  {
    trace("Skipping network configuration checks due to DEV env");
    return;
  }

  if ( -e $suseNetworkConfig && -f $suseNetworkConfig )
  {
    $base  = $suseNetworkConfig;
    $fixed = s_suseCheckNetworkConfig($base);
  }
  elsif( -e $oelNetworkConfig && -f $oelNetworkConfig )
  {
    $base  = $oelNetworkConfig;
    $fixed = s_oelCheckNetworkConfig($base);
  }

  # if we dont need to do any more work then just leave
  if( ! $fixed )
  {
    return;
  }

  # otherwise lets swap the temporary files with the new ones
  $save = $base . $checkNetworkSave;
  $temp = $base . $checkNetworkTemp;

  # cleanup any original files
  unlink $save;

  # move the original base file to a temporary name
  rename $base, $save;

  # then move the temporary file to this new name
  rename $temp, $base;

  # and let the world know what we have done
  trace("Disabled LinkLocal Address manipulation");
}

#------------------------------------------------------------------------------
# Function: check if network configuration files need to be updated to 
#           disable 169.254/16 route usage (ZEROCONF)
#
# Args    : [0]  Base path to OEL configuration files
#------------------------------------------------------------------------------
sub s_oelCheckNetworkConfig
{
  my $baseFile = $_[0];
  my $tempFile = $_[0] . $checkNetworkTemp;
  my $numFound = 0;
  my $found    = FALSE;
  my $line;

  unlink $tempFile;

  open BASEFILE, "<$baseFile";
  open TEMPFILE, ">$tempFile";

  foreach $line (<BASEFILE>)
  {
    chomp $line;
    if ( $line !~ /NOZEROCONF=\S*/ || $line =~ /^\s*#/ )
    {
      print TEMPFILE "$line\n";
      next;
    }

    # track that we found at least one occurrence
    # but dont output it to the file yet
    $numFound++;
  }

  # if we found exactly one occurrence then we are fine
  if( 1 == $numFound )
  {
    $found = TRUE;
  }
  # if we didnt find any or we found more than one then force just one
  else
  {
    $found = FALSE;
    
    # and output the one good line to the end of the file
    print TEMPFILE "NOZEROCONF=yes\n";
  }
  close BASEFILE;
  close TEMPFILE;

  # if we found the setting, then we can just throw away this temp file
  if( $found )
  {
    unlink $tempFile;
  }

  return ! $found;
}

#------------------------------------------------------------------------------
# Function: check if network configuration files need to be updated to 
#           disable 169.254/16 route usage (ZEROCONF)
#
# Args    : [0]  Base path to suse network configuration files
#------------------------------------------------------------------------------
sub s_suseCheckNetworkConfig
{
  my $found    = FALSE;
  my $baseFile = $_[0];
  my $tempFile = $_[0] . $checkNetworkTemp;
  my $line;

  unlink $tempFile;

  open BASEFILE, "<$baseFile";
  open TEMPFILE, ">$tempFile";

  foreach $line (<BASEFILE>)
  {
    chomp $line;
    if ( $line !~ /LINKLOCAL_INTERFACES=\S*/ || $line =~ /^\s*#/ )
    {
      print TEMPFILE "$line\n";
      next;
    }

    # output to the file the same line but stick a comment '#' at the front
    print TEMPFILE "#${line}\n";

    # track that we found at least one occurrence
    # but dont output it to the file yet
    $found = TRUE;
  }

  # if we did not find any broken lines, then we throw away this temp file
  if( ! $found )
  {
    unlink $tempFile;
  }

  return $found;
}

####---------------------------------------------------------
#### Restore init scripts from old CRS home
# ARGS : [0] old CRS home
sub s_restoreInitScripts
{
  my $oldcrshome = $_[0];
  trace("restore init scripts");

  my $initOhasd = 'init.ohasd';
  my $ohasd = 'ohasd';
  my $oldInitOhasd  = catfile ($oldcrshome, 'crs', 'init', $initOhasd);
  my $oldOhasd  = catfile ($oldcrshome, 'crs', 'init', $ohasd);

  my $newInitOhasd  = catfile ($CFG->params('ID'), $initOhasd);
  my $newOhasd  = catfile ($CFG->params('ID'), $ohasd);

  # restore the init scripts from the old home
  copy_file($oldInitOhasd, $newInitOhasd);
  copy_file($oldOhasd, $newOhasd);
}

####---------------------------------------------------------
#### Restore an OLR backup for downgrading
# ARGS : 0
sub s_restoreolrloc
{
  my $oldolr = $CFG->params('OLRCONFIG');
  my $oldolrbkp = $oldolr . ".bkp";

  trace("restore old olr.loc file");
  return copy_file($oldolrbkp, $oldolr);
}

####---------------------------------------------------------
#### Check if an OLR backup exists
# ARGS : 0
sub s_checkolrbackup
{
  my $oldolr = $CFG->params('OLRCONFIG');
  my $oldolrbkp = $oldolr . ".bkp";

  if (! (-e $oldolrbkp))
  {
    trace("Could not find OLR backup");
    return FAILED;
  }

  return SUCCESS;
}

####---------------------------------------------------------
#### Function for restoring ASM files during the downgrade
#### from 12c to 11201/2
# ARGS : 0
sub s_restoreASMFiles
{
  my $optorcldir = $CFG->params('EXTERNAL_ORACLE');
  my $optorclbindir = $CFG->params('EXTERNAL_ORACLE_BIN');
  my $asmgidfile = catfile($optorclbindir, 'setasmgid');
  my $asmgidorg = catfile($CFG->OLD_CRS_HOME, 'bin', 'setasmgid');
  my $ORA_DBA_GROUP = $CFG->params('ORA_DBA_GROUP');
  
  trace("Re-creating the directory '$optorclbindir'"); 
  if (-d $optorcldir)
  {
    mkpath($optorclbindir);
  }
  
  if (!(-d $optorclbindir))
  {
    error("Failed to re-create $optorclbindir");
    return FAILED;
  }
  else
  {
    if (FAILED == s_set_ownergroup($CFG->SUPERUSER, $ORA_DBA_GROUP,
                                    $optorclbindir))
    {
      error("Failed to change the ownership of $optorclbindir");
      return FAILED;
    }

    if (FAILED == s_set_perms("0750", $optorclbindir))
    {
      error("Failed to change the permissions of $optorclbindir");
      return FAILED;
    }
  } 
    
  trace("Restoring $asmgidfile");
  if (FAILED == copy_file($asmgidorg, $asmgidfile,
                           $CFG->SUPERUSER, $ORA_DBA_GROUP))
  {
    error("Failed to restore $asmgidfile");
    return FAILED;
  }

  if (FAILED == s_set_perms("4710", $asmgidfile))
  {
    error("Failed to change permissions on $asmgidfile");
    return FAILED;
  }

  trace("Setting the ownership and permissions of $optorcldir "
       ."to the original");
  if (-d $optorcldir)
  {
    s_set_ownergroup($CFG->SUPERUSER, $ORA_DBA_GROUP, $optorcldir);
    s_set_perms("0750", $optorcldir);
  }

  # Remove /etc/oracle/setasmgid from higher version
  my $file = catfile($CFG->params('OCRCONFIGDIR'), 'setasmgid');
  if (-f $file)
  {
    trace("Remove $file");
    s_remove_file("$file");
  }

  return SUCCESS;
}

# copy afd file to init.d and to rc levels
sub s_copy_afdinit_init
{
  my $INITDIR        = catfile ($CFG->params('ORACLE_HOME'), "crs", "init");
  my $INITDIR_SRV    = catfile ($INITDIR, "afd");

  s_copy_to_initdir($INITDIR_SRV, "afd") || return FAILED;
  s_copy_to_rcdirs ($INITDIR_SRV, "afd") || return FAILED;
  return SUCCESS;
}

sub s_rm_afdinit_init
{
  my $afdfile = catfile($INITD, "afd");  
  trace("Removing /etc/init.d/afd");
  unlink($afdfile);
  return SUCCESS;
}

sub s_rm_afdinit_rclevel
{
  s_clean_rcdirs ("afd");
  return SUCCESS;
}

sub s_osd_setup
{
   #TODO: shmubeen remove later
   my $afdInstall = FALSE;

   $afdInstall = uc($ENV{'ORA_ENABLE_AFD_INSTALL'});
   if ( $afdInstall ne "TRUE" )
   {
     trace("AFD disabled because of ENV in test mode");
     return SUCCESS;
   }
   # Setup afd init files
   s_copy_afdinit_init();
    return SUCCESS;
}

sub s_unregister_service
{
    # TBD
    return SUCCESS;
}

sub s_redirect_souterr
{
# no-op on Linux as we don't want to redirect output
    return SUCCESS;
}
sub s_restore_souterr
{
### no-op on Linux
    return SUCCESS;
}

##====================================================================
sub system {
  return system_cmd(@_);
}

#-------------------------------------------------------------------------------
# Function:  Returns the path to the qosctl script
# Args    :  none
# Returns :  Path to the qosctl script
#-------------------------------------------------------------------------------
sub s_get_qosctl_path {
  my $execPath = catfile( $CFG->ORA_CRS_HOME, 'bin', 'qosctl' );
  if ( !( -x "${execPath}" ) ) {
    die(dieformat(1008, $execPath));
  }
  return $execPath;
}

### TO DO
#- merge s_reset_crshome & s_reset_crshome1
#- dup s_CheckNetworkConfig, s_oelCheckNetworkConfig

# missing
#  my @exp_osd  = qw(
#                    s_deltOldServ
#                    s_removeSCR             s_removeFenceServ
#                    s_copyRegKey            s_stopDeltOldASM
#                    );



1;
